<?php
/**
 * Classe qui a pour objectif de recenser les fonctions utiles pour les images
 * 
 */
class ImageHelper {

    /**
     * Permet de récuperer le base 64 d'une image converti en JPEG en choisissant sa qualité et sa taille max
     * 
     * @param mixed $root : chemin vers le fichier de l'image
     * @param int $qualite : qualité de 0 à 100 (pire qualité à meilleure qualité) (0 par défaut)
     * @param int $maxSize : taille maximum de l'image (600 par défaut)
     * @return bool|string false si l'image n'existe pas, sinon renvoi le base 64 de l'image
     */
    static function getResizedImageJpeg(string $root, int $qualite = 0, int $maxSize = 600) : bool|string
    {
        $image = imagecreatefromstring(file_get_contents($root));

        if ($image !== false) {
            $originalWidth = imagesx($image);
            $originalHeight = imagesy($image);

            $resizeWidth = ($originalWidth > $originalHeight) ? $maxSize : $originalWidth;

            list($newWidth, $newHeight) = ImageHelper::scaleDimensions($originalWidth, $originalHeight, $resizeWidth, null);
            $resizedImage = imagescale($image, $newWidth, $newHeight);

            ob_start();

            imagejpeg($resizedImage, null, $qualite);

            $base64Image = base64_encode(ob_get_clean());

            imagedestroy($resizedImage);

            if (!empty($base64Image)) {
                return $base64Image;
            }
        }

        return false;
    }

    /**
     * Permet d'avoir les dimensions à l'echelle pour une image
     * @param int $width : largeur de l'image
     * @param int $height : hauteur de l'image
     * @param int $maxWidth : largeur max de l'image
     * @param int|null $maxHeight : hauteur max de l'image
     * @return array : les valeurs finales de l'image à l'echelle
     */
    static function scaleDimensions(int $width, int $height, int $maxWidth, int|null $maxHeight) : array
    {
        if (!$maxWidth && !$maxHeight) {
            return [$width, $height];
        }

        $aspectRatio = $width / $height;

        if ($maxWidth && (!$maxHeight || ($maxWidth / $aspectRatio <= $maxHeight))) {
            return [$maxWidth, $maxWidth / $aspectRatio];
        } elseif ($maxHeight && (!$maxWidth || ($maxHeight * $aspectRatio <= $maxWidth))) {
            return [$maxHeight * $aspectRatio, $maxHeight];
        }

        return [$width, $height];
    }
}