<?php
namespace ODE_API\V8\Service;

use ODE_API\V8\BeanDecorator\BeanManager;
use ODE_API\V8\JsonApi\Helper\AttributeObjectHelper;
use ODE_API\V8\JsonApi\Helper\PaginationObjectHelper;
use ODE_API\V8\JsonApi\Helper\RelationshipObjectHelper;
use ODE_API\V8\JsonApi\Response\LinksResponse;
use ODE_API\V8\JsonApi\Response\DataResponse;
use ODE_API\V8\JsonApi\Response\DocumentResponse;
use ODE_API\V8\JsonApi\Response\MetaResponse;

use ODE_API\V8\Param\GetOAuthIdentityParams;
use ODE_API\V8\Param\GetOAuthIndividuParams;
use ODE_API\V8\Param\SetOAuthIndividuParams;
use ODE_API\V8\Param\SetOAuthHistoryCnxParams;

use ODE\Mailer\OdeEmail;
use ODE\Mailer\OdeMailer;

use Exception;
use Slim\Http\Request;


class OAuthCRMService
{
    /**
     * @var BeanManager
     */
    protected $beanManager;
    
    /**
     * @var AttributeObjectHelper
     */
    protected $attributeHelper;
    
    /**
     * @var RelationshipObjectHelper
     */
    protected $relationshipHelper;
    
    /**
     * @var PaginationObjectHelper
     */
    protected $paginationHelper;
    
    /**
     * @param BeanManager $beanManager
     * @param AttributeObjectHelper $attributeHelper
     * @param RelationshipObjectHelper $relationshipHelper
     * @param PaginationObjectHelper $paginationHelper
     */
    public function __construct(BeanManager $beanManager, AttributeObjectHelper $attributeHelper, RelationshipObjectHelper $relationshipHelper, PaginationObjectHelper $paginationHelper)
    {
        $this->beanManager        = $beanManager;
        $this->attributeHelper    = $attributeHelper;
        $this->relationshipHelper = $relationshipHelper;
        $this->paginationHelper   = $paginationHelper;
    }
    
    /**
     * getIdentity
     * @param GetOAuthIdentityParams $params
     * @return DocumentResponse
     * @throws AccessDeniedException
     */
    public function getIdentity(GetOAuthIdentityParams $params, Request $request)
    {
        global $sugar_config, $db;

        $email_identity = $params->getEmail();
        $module         = 'OPS_individu';
        
        $bean = \BeanFactory::newBean($module);

        if (!$bean instanceof \SugarBean) {
            throw new \InvalidArgumentException(sprintf('Module %s does not exist', $module));
        }





        // Format the query properly...
        $result = $db->query(
            sprintf(
                'select %1$s.id, %1$s.canal, %1$s.statut_compte, %1$s.connexion_internet from %1$s
                    inner join email_addr_bean_rel on email_addr_bean_rel.bean_id = %1$s.id and email_addr_bean_rel.deleted = 0
                    inner join email_addresses on email_addresses.id = email_addr_bean_rel.email_address_id and email_addresses.deleted = 0
                    where email_addresses.email_address like \'%2$s\'
                        and email_addr_bean_rel.primary_address = 1
                        and email_addresses.deleted = 0
                        and ( ( %1$s.statut_compte = "actif" and %1$s.connexion_internet = "1" )  
                        OR ( %1$s.statut_compte = "inactif" and %1$s.canal = "internet" ) )',
                \BeanFactory::newBean($module)->getTableName(),
                $email_identity
            )
        );

        // Fetch rows (Loop association)
        $row_results = array();
        while($row = $db->fetchRow($result)){ 
            $row_results[$row['id']] = $row; 
        }

        // Format output.
        $data = empty($row_results)? false : array( 'identity' => $row_results );



        // Historisation Connexion
        if( !empty($row_results) && sizeof($row_results) > 0 ){

            $id_individu = array_values($row_results)[0]['id']; // Ok, Record ID de OPS_individu

            $individu_Bean = \BeanFactory::getBean( $module, $id_individu );
            $individu_Bean->load_relationship( 'ops_historique_connexion_ops_individu' );

            $historique_Bean = \BeanFactory::newBean( 'OPS_historique_connexion' );

            $historique_Bean->derniere_connexion = date('Y-m-d H:i:s');
            $historique_Bean->derniere_connexion_info = '-- en cours --';
            $historique_id = $historique_Bean->save();
            
            if( $historique_id === false ){
                $GLOBALS['log']->fatal("OAuthCRMService :: OPS_historique_connexion [Historisation de la connexion du ".date('Y-m-d H:i:s')." impossible]");
            }else{
                $individu_Bean->ops_historique_connexion_ops_individu->add( $historique_Bean );

                // Récupère l'id de l'historique pour le retourner et l'utiliser afin de laisser la trace d'information de connexion (browser details).
                $data['historique_id'] = $historique_id;
            }
    
        

        }

        
        $response = new DocumentResponse(); // TODO : Use IndividuResponse
        $response->setData($data);



        return $response;
    }



    
    /**
     * getIndividu
     * @param GetOAuthIndividuParams $params
     * @return DocumentResponse
     * @throws AccessDeniedException
     */
    public function getIndividu(GetOAuthIndividuParams $params, Request $request)
    {
        global $sugar_config;

        $id_individu    = $params->getId();
        $module         = 'OPS_individu';
        

        $bean = \BeanFactory::newBean($module);

        if (!$bean instanceof \SugarBean) {
            throw new \InvalidArgumentException(sprintf('Module %s does not exist', $module));
        }


        try {
            $obj_module = $this->beanManager->getBeanSafe($module, $id_individu);
        } catch (\Exception $exception) {
            throw new Exception('Usager inconnu', 401);
        }


        $data = false;  // Default failover for data response.
        if( !empty( $obj_module ) ){
            
            # Temporaire : Anomalie #1609
            $obj_module->canal  = isset( $obj_module->canal )? $obj_module->canal : 'internet';
            $obj_module->statut = isset( $obj_module->statut )? $obj_module->statut : 'actif';

            $data = [
                'individu' => [
                    'id'                  => $obj_module->id,
                    'name'                => $obj_module->name,
                    'modified_user_id'    => $obj_module->modified_user_id,
                    'description'         => $obj_module->description,
                    'deleted'             => $obj_module->deleted,
                    'salutation'          => $obj_module->salutation,
                    'first_name'          => $obj_module->first_name,
                    'last_name'           => $obj_module->last_name,
                    'full_name'           => $obj_module->full_name,
                    'title'               => $obj_module->title,
                    'email1'              => $obj_module->email1,
                    'nationalite'         => $obj_module->nationalite,
                    'situation_familiale' => $obj_module->situation_familiale,
                    'statut_compte'      => $obj_module->statut_compte,
                    'canal'              => $obj_module->canal,
                    'connexion_internet' => $obj_module->connexion_internet,
                ]
            ];
        }

        $response = new DocumentResponse();
        $response->setData($data);

        return $response;
    }


    /**
     * setIndividu
     * @param SetOAuthIndividuParams $params
     * @return DocumentResponse
     * @throws AccessDeniedException
     */
    public function setIndividu(SetOAuthIndividuParams $params, Request $request)
    {
        global $sugar_config;

        $info_individu  = $params->getData();
        $module         = 'OPS_individu';


        $bean = \BeanFactory::newBean($module);

        if (!$bean instanceof \SugarBean) {
            throw new \InvalidArgumentException(sprintf('Module %s does not exist', $module));
        }


        try {
            $obj_module = $this->beanManager->newBeanSafe($module);
        } catch (\Exception $exception) {
            throw new Exception('Créa inconnu', 401);
        }


        if (!$obj_module->ACLAccess('save')) {
            throw new AccessDeniedException();
        }


        $obj_module->name               = $info_individu["name"];
        $obj_module->deleted            = $info_individu["deleted"];
        $obj_module->first_name         = $info_individu["first_name"];
        $obj_module->last_name          = $info_individu["last_name"];
        $obj_module->full_name          = $info_individu["full_name"];
        $obj_module->email1             = $info_individu["email1"];
        $obj_module->statut_compte      = $info_individu["statut_compte"];
        $obj_module->canal              = $info_individu["canal"];
        $obj_module->type_individu      = "adulte";
        $obj_module->connexion_internet = $info_individu["connexion_internet"];

        $id = $obj_module->save();

        $data = false;  // Default failover for data response.
        if( !empty($id) && $obj_module->id == $id ){

            $data = [
                'individu' => [
                    'id'                  => $obj_module->id,
                    'name'                => $obj_module->name,
                    'modified_user_id'    => $obj_module->modified_user_id,
                    'description'         => $obj_module->description,
                    'deleted'             => $obj_module->deleted,
                    'salutation'          => $obj_module->salutation,
                    'first_name'          => $obj_module->first_name,
                    'last_name'           => $obj_module->last_name,
                    'full_name'           => $obj_module->full_name,
                    'title'               => $obj_module->title,
                    'email1'              => $obj_module->email1,
                    'nationalite'         => $obj_module->nationalite,
                    'situation_familiale' => $obj_module->situation_familiale,
                    'statut_compte'      => $obj_module->statut_compte,
                    'canal'              => $obj_module->canal,
                    'connexion_internet' => $obj_module->connexion_internet,
                ]
            ];
        }


        $response = new DocumentResponse();
        $response->setData($data);

        return $response;
    }


    /**
     * updateIndividu
     * @param SetOAuthIndividuParams $params
     * @return DocumentResponse
     * @throws AccessDeniedException
     */
    public function updateIndividu(SetOAuthIndividuParams $params, Request $request)
    {
        global $sugar_config;

        $info_individu  = $params->getData();
        $module         = 'OPS_individu';

        $bean = \BeanFactory::newBean($module);

        if (!$bean instanceof \SugarBean) {
            throw new \InvalidArgumentException(sprintf('Module %s does not exist', $module));
        }

        $data = false; // Default failover for data response.
        $response = new DocumentResponse();

        // Il nous faut au moins un UID pour cet OPS_individu !
        if( !isset($info_individu["id"]) ) {

            $response->setData($data);
            $response->setMeta(new MetaResponse(
                [
                    'message' => sprintf(
                        'L\'UID de %s est manquant, le paramètre « id » est requis !',
                        $module
                    )
                ]
            ));

            return $response;

        }

        $id_individu = $info_individu["id"];

        try {
            $obj_module = $this->beanManager->getBeanSafe($module, $id_individu);
        } catch (\Exception $exception) {
            throw new Exception('Usager inconnu', 401);
        }



        if (!$obj_module->ACLAccess('save')) {
            throw new AccessDeniedException();
        }


        // Bulk mode anonyme with an override of the given parameters (args subimted from /update route).
        unset( $info_individu["id"] );
        foreach ($info_individu as $key => $value) {
            $obj_module->{$key} = $value;
        }

        // Save avec contrôle de retour.
        $id = $obj_module->save();

        if( !empty($id) && $id_individu == $id ){

            $data = [
                'individu' => [
                    'id'                  => $obj_module->id,
                    'name'                => $obj_module->name,
                    'modified_user_id'    => $obj_module->modified_user_id,
                    'description'         => $obj_module->description,
                    'deleted'             => $obj_module->deleted,
                    'salutation'          => $obj_module->salutation,
                    'first_name'          => $obj_module->first_name,
                    'last_name'           => $obj_module->last_name,
                    'full_name'           => $obj_module->full_name,
                    'title'               => $obj_module->title,
                    'email1'              => $obj_module->email1,
                    'nationalite'         => $obj_module->nationalite,
                    'situation_familiale' => $obj_module->situation_familiale,
                    'statut_compte'      => $obj_module->statut_compte,
                    'canal'              => $obj_module->canal,
                    'connexion_internet' => $obj_module->connexion_internet,

                ]
            ];
        }else{
            $response->setMeta(new MetaResponse(
                [
                    'message' => sprintf(
                        'La mise à jour de %s ne s\'est pas faite corretement pour l\UID id: %s !',
                        $module,
                        $id_individu
                    )
                ]
            ));

        }


        $response->setData($data);

        return $response;
    }



    
    /**
     * updateHistoryConnexion
     * @param SetOAuthHistoryCnxParams $params
     * @return DocumentResponse
     * @throws AccessDeniedException
     */
    public function updateHistoryConnexion(SetOAuthHistoryCnxParams $params, Request $request)
    {
        global $sugar_config;

        $info_history_cnx  = $params->getData();
        $module            = 'OPS_historique_connexion';

        $bean = \BeanFactory::newBean($module);

        if (!$bean instanceof \SugarBean) {
            throw new \InvalidArgumentException(sprintf('Module %s does not exist', $module));
        }

        $data = false; // Default failover for data response.
        $response = new DocumentResponse(); // Use the DocumentResponse - it's the same logic.


        # Integrity and security control.

        // Il nous faut au moins un UID pour cet OPS_historique_connexion !
        if( !isset($info_history_cnx["historique_id"]) ) {

            $response->setData($data);
            $response->setMeta(new MetaResponse(
                [
                    'message' => sprintf(
                        'L\'UUID de %s est manquant, le paramètre « historique_id » est requis !',
                        $module
                    )
                ]
            ));

            return $response;

        }

        // Il nous faut au moins l'UID pour l'usager en lien avec ce history !
        if( !isset($info_history_cnx["id"]) ) {

            $response->setData($data);
            $response->setMeta(new MetaResponse(
                [
                    'message' => sprintf(
                        'L\'UUID de %s est manquant, le paramètre « id » est requis !',
                        'OPS_individu'
                    )
                ]
            ));

            return $response;

        }

        
        // Envoi de la notification de connexion à l'usager
        $idTemplateMail = $sugar_config['opensocle']['notif_usager_connexion'] ?? "";


        // Il nous faut au moins des information sur le browser / device
        if( !isset($info_history_cnx["browser_info"]) ) {

            $response->setData($data);
            $response->setMeta(new MetaResponse(
                [
                    'message' => 'Les informations de connexion sont manquantes, le paramètre « browser_info » est requis !'
                ]
            ));

            return $response;

        }

        // Intégrité...

        preg_match( '/^(\d+|[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12})$/i', trim($info_history_cnx["historique_id"]), $matches_histo_id, PREG_OFFSET_CAPTURE);
        preg_match( '/^(\d+|[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12})$/i', trim($info_history_cnx["id"]), $matches_id, PREG_OFFSET_CAPTURE);
        if( empty($matches_histo_id) || empty($matches_id) ){

            $response->setData($data);
            $response->setMeta(new MetaResponse(
                [
                    'message' => 'Le paramètre UUID est incorrect'
                ]
            ));

            return $response;

        }

        $id_history_cnx = trim( $info_history_cnx["historique_id"] );
        $id_individu    = trim( $info_history_cnx["id"] );
        $browser_info   = trim( strip_tags($info_history_cnx["browser_info"]) ) ;


        # More security...
        $individu_Bean = \BeanFactory::getBean( 'OPS_individu', $id_individu );

        if(!empty($idTemplateMail)) {
            $mailer = new OdeMailer();

            $ode_email = new OdeEmail([
                'bean_source_id' => $id_individu,
                'bean_source_name' => 'OPS_individu',
                'bean_historisation_id' => $id_individu,
                'bean_historisation_name' => 'OPS_individu',
                'email_template_id' => $idTemplateMail,
                'dest_to' => $individu_Bean->email1,
            ]);

            $ode_email->body_html = str_replace('$navigateur', $browser_info, $ode_email->body_html);
            $ode_email->body = str_replace('$navigateur', $browser_info, $ode_email->body);


            $mailer->send($ode_email);

        }

        // Individu have this cnx in relationship ?
        try {
            $historique_Bean = \BeanFactory::getBean( $module, $id_history_cnx );
        } catch (\Exception $exception) {
            throw new Exception('Historique de connexion inconnu', 401);
        }


        if (!$historique_Bean->ACLAccess('save')) {
            throw new AccessDeniedException();
        }

        // Update avec Save et contrôle de retour.
        $historique_Bean->derniere_connexion_info = $browser_info;
        $historique_id = $historique_Bean->save();

        $data = [
            'id' => $id_individu,
        ];

        if( !empty($historique_id) && $id_history_cnx == $historique_id ){

            $data['historique_id'] = $historique_id;
            $data['message'] = 'La mise à jour des informations de connexion à réussi.';

            $response->setMeta(new MetaResponse(
                [
                    'message' => $data['message']
                ]
            ));

        }else{

            $data['historique_id'] = 'failed';
            $data['message'] = 'La mise à jour des informations de connexion à échoué.';

            $response->setMeta(new MetaResponse(
                [
                    'message' => $data['message']
                ]
            ));
        }


        $response->setData($data);

        return $response;
    }


    
}
