<?php

namespace ODE_API\V8\Service;

use ODE_API\V8\BeanDecorator\BeanManager;
use ODE_API\V8\JsonApi\Helper\AttributeObjectHelper;
use ODE_API\V8\JsonApi\Helper\PaginationObjectHelper;
use ODE_API\V8\JsonApi\Helper\RelationshipObjectHelper;
use ODE_API\V8\JsonApi\Response\LinksResponse;
use ODE_API\V8\JsonApi\Response\DataResponse;
use ODE_API\V8\JsonApi\Response\DossierResponse;
use ODE_API\V8\JsonApi\Response\MetaResponse;

use ODE_API\V8\Param\DossierSetParams;

use ODE\Model\AppairageModel;
use ODE\Model\IndividuModel;
use ODE\Model\PersonneMoraleModel;
use ODE\Model\DossierModel;

use BeanFactory;
use OPS_appairage;

use Exception;
use Slim\Http\Request;


class AppairageService
{
    /**
     * @var BeanManager
     */
    protected $beanManager;

    /**
     * @var AttributeObjectHelper
     */
    protected $attributeHelper;

    /**
     * @var RelationshipObjectHelper
     */
    protected $relationshipHelper;

    /**
     * @var PaginationObjectHelper
     */
    protected $paginationHelper;

    /**
     * @param BeanManager $beanManager
     * @param AttributeObjectHelper $attributeHelper
     * @param RelationshipObjectHelper $relationshipHelper
     * @param PaginationObjectHelper $paginationHelper
     */
    public function __construct(BeanManager $beanManager, AttributeObjectHelper $attributeHelper, RelationshipObjectHelper $relationshipHelper, PaginationObjectHelper $paginationHelper)
    {
        $this->beanManager        = $beanManager;
        $this->attributeHelper    = $attributeHelper;
        $this->relationshipHelper = $relationshipHelper;
        $this->paginationHelper   = $paginationHelper;
    }

    /**
     * createDossier
     * @param DossierSetParams $params
     * @param $path
     * @return DossierResponse
     * @throws AccessDeniedException
     */
    public function createAppairage(DossierSetParams $params, Request $request)
    {
        // On récupere les données
        $donnees = $params->getData();

        // On vérifie que les données ne sont pas vide 
        if (!is_array($donnees) || count($donnees) === 0) {
            throw new Exception('Données vide', 401);
        }

        // On vérifie que l'id de l'objet à appairer n'est pas vide
        if (empty($donnees["object_id"])) {
            throw new Exception("L'id du objet 'object_id' à appairer est obligatoire", 401);
        }

        // On vérifie que le type de l'objet à appairer n'est pas vide
        if (empty($donnees["object_type"])) {
            throw new Exception("Le type de l'objet 'object_type' à appairer est obligatoire", 401);
        }

        // On vérifie que le type de l'objet à appairer est égal à une des valeurs du tableau
        $object_types = ["OPS_individu", "OPS_dossier", "OPS_personne_morale"];
        if (!in_array($donnees["object_type"], $object_types)) {
            throw new Exception("Le type de l'objet 'object_type' doit etre égal à une des valeurs : OPS_individu, OPS_personne_morale ou OPS_dossier", 401);
        }

        // On vérifie que le le code d'appairage n'est pas vide
        if (empty($donnees["appairage_name"])) {
            throw new Exception("Le code d'appairage 'appairage_name' est obligatoire", 401);
        }

        // On vérifie que le le code d'appairage n'est pas vide
        if (empty($donnees["appairage_logiciel"])) {
            throw new Exception("Le logiciel d'appairage 'appairage_logiciel' est obligatoire", 401);
        }

        // On vérifie que le type de l'objet à appairer est égal à une des valeurs du tableau
        $appairage_logiciels = ["gestionfin", "cde", "keycloak"];
        if (!in_array($donnees["appairage_logiciel"], $appairage_logiciels)) {
            throw new Exception("Le logiciel d'appairage 'appairage_logiciel' doit etre égal à une des valeurs : gestionfin, cde ou keycloak", 401);
        }

        // On vérifie si on a un statut d'appairage, par défaut il est égal à "ok" 
        $donnees["appairage_statut"] =  (!empty($donnees["appairage_statut"]) && $donnees["appairage_statut"] === "err") ? "err" : "ok";

        // On vérifie si on a une description pour l'appairage
        $donnees["appairage_description"] =  (!empty($donnees["appairage_description"])) ? $donnees["appairage_description"] : "";


        # Traitement de la  façon dont l'ppairage est soit créé, soiyt mis à jour...
        # On récupère l'appairage ou une liste d'appairages, et auquel cas on préserve le plus récent et on retire les anciens.
        # Ce sera le plus recent donc qui sera mis à jour vias l'id, soit on le cré simplement.
        $appairages = AppairageModel::getByObject($donnees["object_id"], $donnees["object_type"], $donnees["appairage_logiciel"], $donnees["appairage_statut"]);

        // Traitement de tri multi-dimentions
        $columns = array_column($appairages, 'date_modified');
        array_multisort($columns, SORT_DESC, $appairages);
       

        // Récup dernier id d'appairage...
        if( is_array($appairages) && sizeof($appairages) > 0 ){
            $last_appairage = array_shift( $appairages );
            $last_appairage_id = $last_appairage['id'];

            foreach( $appairages as $index => $appairage ){
                $old_appairage_id = $appairage['id'];

                $obj_appairage = \BeanFactory::getBean( 'OPS_appairage', $old_appairage_id );
                $obj_appairage->mark_deleted($old_appairage_id);

            }

            // Update.
            $obj_appairage = \BeanFactory::getBean( 'OPS_appairage', $last_appairage_id );
            $obj_appairage->appairage_description = $donnees["appairage_description"];
            $obj_appairage->name = $donnees["appairage_name"];

            $appairage_id = $obj_appairage->save();

        }
        // Création d'un appairage purement et simplement...
        else{

            $appairage_id = OPS_appairage::createAppairage($donnees);

        }




        // Si l'id est vide on retourne une erreur
        if (empty($appairage_id)) {
            throw new Exception("Erreur de création de l'appairage", 401);
        }

        // On retourne l'id du dossier
        $response = new DossierResponse();
        $response->setData(["appairage_id" => $appairage_id]);

        return $response;
    }

    /**
     * getAppairageByObject
     * @param DossierSetParams $params
     * @param $path
     * @return DossierResponse
     * @throws AccessDeniedException
     */
    public function getAppairageByObject(DossierSetParams $params, Request $request)
    {
        // On récupere les données
        $donnees = $params->getData();

        // On vérifie que l'id de l'objet à appairer n'est pas vide
        if (empty($donnees["object_id"])) {
            throw new Exception("L'id du objet 'object_id' à appairer est obligatoire", 401);
        }

        // On vérifie que le type de l'objet à appairer n'est pas vide
        if (empty($donnees["object_type"])) {
            throw new Exception("Le type de l'objet 'object_type' à appairer est obligatoire", 401);
        }

        // On vérifie que le type de l'objet à appairer est égal à une des valeurs du tableau
        $object_types = ["OPS_individu", "OPS_dossier", "OPS_personne_morale"];
        if (!in_array($donnees["object_type"], $object_types)) {
            throw new Exception("Le type de l'objet 'object_type' doit etre égal à une des valeurs : OPS_individu, OPS_personne_morale ou OPS_dossier", 401);
        }

        // On vérifie que l'object existe en base données
        if ($this->isObjectExist($donnees["object_id"], $donnees["object_type"]) === false) {
            throw new Exception("L'objet ( type = '" . $donnees["object_type"] . "' et id = '" . $donnees["object_id"] . "' ) n'a pas pu etre récupéré", 401);
        }

        // On vérifie que le logiciel de l'objet à appairer est égal à une des valeurs du tableau
        $appairage_logiciels = ["gestionfin", "cde", "keycloak"];
        if (!empty($donnees["appairage_logiciel"]) && !in_array($donnees["appairage_logiciel"], $appairage_logiciels)) {
            throw new Exception("Le logiciel d'appairage 'appairage_logiciel' peut etre vide ou égal à une des valeurs : gestionfin, cde ou keycloak", 401);
        }

        // Si le logiciel de l'objet à appairer n'existe pas on l'initialise à vide
        if (!isset($donnees["appairage_logiciel"])) {
            $donnees["appairage_logiciel"] = "";
        }

        // On vérifie si on a un statut d'appairage, si different de "ok" ou "err" on l'initialise à vide pour retourner l'ensemble des appairages
        $donnees["appairage_statut"] =  (!empty($donnees["appairage_statut"]) && in_array($donnees["appairage_statut"], ["ok", "err"])) ? $donnees["appairage_statut"] : "";

        $appairages = AppairageModel::getByObject($donnees["object_id"], $donnees["object_type"], $donnees["appairage_logiciel"], $donnees["appairage_statut"]);

        // On retourne l'id du dossier
        $response = new DossierResponse();
        $response->setData($appairages);

        return $response;
    }


    /**
     * isObjectExist
     * @param $id
     * @param $type
     * @return boolean
     */
    public function isObjectExist($id, $type)
    {

        $model = false;

        if ($type === "OPS_individu")  $model = new IndividuModel($id);
        if ($type === "OPS_personne_morale")  $model = new PersonneMoraleModel($id);
        if ($type === "OPS_dossier")  $model = new DossierModel($id);

        if ($model) {
            $object = $model->getBean();
        }

        return (is_array($object) && count($object) > 0) ? true : false;
    }
}
