<?php

namespace ODE\Helper;

use BeanFactory;

use ODE\Model\GenerateurReferentielModel;
use ODE\Helper\OdeDateHelper;
use ODE\Generateur\Factory\OdeFieldFactory;

use Datetime;
use DateTimeZone;
use ODE\Model\DispositifModel;

/**
 * Class ReplaceVariable
 *
 * @package ODE\Helper
 */
class ReplaceVariable
{

    /** @var bool|object */
    protected $bean = false;

    private $replace_variable_path;
    private $replace_variable_namespace;
    private $replace_variables;

    /**
     * ReplaceVariable constructor.
     * @param object $bean : Bean SuiteCRM
     */
    public function __construct($bean)
    {
        global $app_list_strings;

        if (empty($bean->id) || empty($bean->object_name)) {
            $GLOBALS['log']->fatal("ReplaceVariable::__construct => L'id ou object_name du bean est vide");
            return;
        }
        
        $this->bean = $bean;

        $this->replace_variable_path = "custom/include/Ode/Helper/ReplaceVariable/";
        $this->replace_variable_namespace = 'ODE\\Helper\\ReplaceVariable\\';
        $this->replace_variables = $this->getReplaceVariables();
    }

    /**
     * @access public
     * @name getReplaceVariables()
     * Fonction qui retourne la liste des replaces variables configurés sur le serveur 
     * 
     * 
     *  @return array       - $replace_variables
     */
    public function getReplaceVariables()
    {
        $replace_variables = [];

        if (is_dir($this->replace_variable_path)) 
        {
            $files = array_diff(scandir($this->replace_variable_path), ['.', '..']);
            foreach ($files as $file_name) 
            {
                if( end( explode('.', $file_name) ) == 'php')
                {
                    $class_name = $this->replace_variable_namespace . str_replace('.php', '', $file_name);
                    $replace_variables[strtolower(str_replace('ReplaceVariable.php', '', $file_name))] = new $class_name($this->bean);
                }
            }
        } 
        else 
        {
            $GLOBALS['log']->fatal(" ReplaceVariable :: getReplaceVariables => Erreur initialisation des replaces variables, action_path (" . $this->replace_variable_path . ") introuvable");
        }
        return $replace_variables;
    }

    /** Fonction qui remplace les variables dans un text à partir du module source d'initialisation
     * @param string    $text 
     * 
     * @return string
     */
    public function replace($text = '',$document_id = '')
    {
        # On vérifie que la classe est bien intitialiser et que le text n'est pas vide
        $text = !empty($text) && gettype($text) === 'string' ? $text : '';
        if (empty($text) || $this->bean === false) return '';

        # On remplace les variables selon le bean source
        if (is_array($this->replace_variables) && count($this->replace_variables) > 0 && array_key_exists(strtolower($this->bean->object_name), $this->replace_variables)) 
        {
            return $this->replace_variables[strtolower($this->bean->object_name)]->remplaceChamps(text:$text,document_id:$document_id);
        }
        else
        {
            return $text;
        }
    }

    /**
     * Fonction qui retourne les noms des champs à utiliser
     * @param object $bean : Bean SuiteCRM
     * 
     * @return array
     */
    public static function getBeanFields($bean = false)
    {
        # Si la classe est mal initialisé, on retourne un tableau vide
        if (empty($bean->id) || empty($bean->field_defs)) return [];

        # On récupére la liste des champs du bean
        $fields = [];
        foreach ($bean->field_defs as $field) {
            $field_name = !empty($field['name']) && $field['name'] !== 'email' ?  $field['name'] : '';
            if (!empty($field_name) && isset($bean->$field_name) && !is_a($bean->$field_name, 'Link2') &&  $field['type'] !== 'link') {
                $fields[] = $field_name;
            }
        }    

        # On fait un tri décroissant ici aussi sur la longueur des clés. (donc une comparaison sur >)
        uasort($fields, function ($field_1, $field_2) {
            return strlen($field_2) > strlen($field_1);
        });

        return $fields;
    }

    /**
     * Fonction qui retourne les champs de type liste
     * @param object $bean : Bean SuiteCRM
     * 
     * @return array
     */
    public static function getBeanFieldsTypeListe($bean = false)
    {
        # Si la classe est mal initialisé, on retourne un tableau vide
        if (empty($bean->id) || empty($bean->field_defs)) return [];

        # On récupére la liste des champs du bean
        $fields = [];
        foreach ($bean->field_defs as $field) {
            $field_name = !empty($field['name']) ?  $field['name'] : '';
            if (!empty($field_name) && $field['type'] === 'enum' && !empty($field['options'])) {
                $fields[$field_name] = $field['options'];
            }
        }    

        return $fields;
    }

    /**
     * Fonction qui retourne la valeur de la liste à partir de la clé
     * @param string $liste_name : Le nom de la liste app_list_strings
     * @param string $value : La valeur du champ en BDD
     * 
     * @return string
     */
    public static function getValueFromListe($liste_name = '', $value = '')
    {
        global $app_list_strings;    
        if ( empty($liste_name) || empty($value) || !array_key_exists($liste_name, $app_list_strings) ) return $value;
        $liste = $app_list_strings[$liste_name];
        return array_key_exists($value, $liste) && !empty($liste[$value]) ? $liste[$value] : $value;
    }

    /**
     * Fonction qui formate la date selon la configuration du user
     * @param string $date
     * 
     * @return string
     */
    public static function formatDate($date = '')
    {
        global $timedate, $current_user;
  
        if (empty($date)) return $date;

        # cas des WS ou le current_user n'est pas correctement initialisé
        if (empty($current_user->id)) $current_user->getSystemUser();

        $UserPrefGmtOffset = $current_user->getUserDateTimePreferences()['userGmtOffset'];
            
        $date = substr( $date , 0, 16);

        if( strpos($date , ' ') === false )
        {
            $format_convertion =  'Y-m-d';
            $format_sortie =  'd/m/Y';
            if( strpos( $date, '-' ) === false  )
            {
                # Récupération du format de date provenant du user
                $format = explode(' ',$GLOBALS['timedate']->get_date_time_format())[0];
                if (empty($format))
                {
                    require_once('modules/Configurator/Configurator.php');

                    $configuratorObj = new Configurator();

                    $format = $configuratorObj->config['default_date_format'];
                }
            }
            else
            {
                $format = 'Y-m-d';
            }
        }
        else
        {
            $format_convertion =  'Y-m-d H:i';
            $format_sortie =  'd/m/Y H:i';
            if( strpos( $date, '-' ) === false  )
            {
                # Récupération du format de date provenant du user
                $format = (!empty($GLOBALS['timedate']->get_date_time_format())) ? $GLOBALS['timedate']->get_date_time_format() : 'd/m/Y H:i';
            }
            else
            {
                $format = 'Y-m-d H:i';
                // Computing the real UTC DateTime based on the TimeZone offset in seconds.
                if( $UserPrefGmtOffset > 0 ){
                    $date_now_synced = date($format, (strtotime($date) + abs($UserPrefGmtOffset*60) ) ); // Real UTC Time with offset in seconds.
                }else{
                    $date_now_synced = date($format, (strtotime($date) - abs($UserPrefGmtOffset*60) ) ); // Real UTC Time without offset in seconds.
                }
            }
        }

        $date_formate = DateTime::createFromFormat( $format, $date );
        $date_formate = $date_formate->format($format_sortie );

        return $date_formate;
    }
}
