<?php

use SuiteCRM\Exception\Exception;
use SuiteCRM\Search\SearchEngine;
use SuiteCRM\Search\SearchQuery;
use SuiteCRM\Search\SearchResults;
use SuiteCRM\Search\SearchModules;
use SuiteCRM\Search\UI\SearchResultsController;

if (!defined('sugarEntry') || !sugarEntry) {
    die('Not A Valid Entry Point');
}

class OpenCRM extends SearchEngine
{
    /* path to search form */
    public $searchFormPath = 'include/SearchForm/SearchForm2.php';

    /*search form class name*/
    public $searchFormClass = 'SearchForm';

    /**
     * Search function run when user goes to Show All and runs a search again.  This outputs the search results
     * calling upon the various listview display functions for each module searched on.
     *
     * @param SearchQuery $query
     *
     * @return SearchResults
     * @throws Exception
     */
    public function search(SearchQuery $query): SearchResults
    {
        $modulesToSearch = $this->getSearchModules();

        $start = microtime(true);

        $results = $this->searchModules($modulesToSearch, $query->getSearchString());

        $end = microtime(true);
        $elapsed = $end - $start;
        $totalHits = 0;

        foreach ($results['modules'] as $moduleHit) {
            $totalHits += count($moduleHit);
        }

        $searchResults = new SearchResults($results['modules'], true, $elapsed, $totalHits);

        return $searchResults;
    }

    public function displayResults(SearchQuery $query, SearchResults $results): void
    {
        // Override the displayResults() method to change how the `SearchWrapper` will show the results.

        // I have used an anonymous class just for simplicity in the example.
        // The extended controller can be normally placed in a separate file.
        $controller = new class ($query, $results) extends SearchResultsController {
            public function __construct(SearchQuery $query, SearchResults $results)
            {
                parent::__construct($query, $results);

                // Let us load a custom template file.
                $this->view->setTemplateFile(__DIR__ . '/OpenCRM.tpl');

            }
        };

        // Finally ask the Controller to render the view.
        $controller->display();
    }

    /**
     * @return array
     */
    protected function getSearchModules(): array
    {
        $unifiedSearchModuleDisplay = SearchModules::getUnifiedSearchModulesDisplay();

        require_once 'include/ListView/ListViewSmarty.php';

        global $beanList, $current_user;

        $users_modules = $current_user->getPreference('globalSearch', 'search');
        $modulesToSearch = [];

        if (!empty($users_modules)) {
            // Use user's previous selections
            foreach ($users_modules as $key => $value) {
                if (isset($unifiedSearchModuleDisplay[$key]) && !empty($unifiedSearchModuleDisplay[$key]['visible'])) {
                    $modulesToSearch[$key] = $beanList[$key];
                }
            }
        } else {
            foreach ($unifiedSearchModuleDisplay as $module => $data) {
                if (!empty($data['visible'])) {
                    $modulesToSearch[$module] = $beanList[$module];
                }
            }
        }

        $current_user->setPreference('globalSearch', $modulesToSearch, 'search');

        return $modulesToSearch;
    }

    /**
     *
     * @param array $modulesToSearch
     * @param string $searchQuery
     * @return array
     * @noinspection DisconnectedForeachInstructionInspection
     * @noinspection PhpIncludeInspection
     */
    private function searchModules(array $modulesToSearch, string $searchQuery): array
    {
        global $beanFiles;

        $unifiedSearchModules = SearchModules::getUnifiedSearchModules();

        $moduleResults = [];
        $moduleCounts = [];
        $listViewDefs = [];

        if (!empty($searchQuery)) {
            foreach ($modulesToSearch as $moduleName => $beanName) {
                require_once $beanFiles[$beanName];
                $seed = new $beanName();

                $listViewData = new ListViewData();

                // Retrieve the original list view defs and store for processing in case of custom layout changes
                require(__DIR__ . '/../../../modules/' . $seed->module_dir . '/metadata/listviewdefs.php');
                $origListViewDefs = $listViewDefs;

                if (file_exists('custom/modules/' . $seed->module_dir . '/metadata/listviewdefs.php')) {
                    require('custom/modules/' . $seed->module_dir . '/metadata/listviewdefs.php');
                }

                if (!isset($listViewDefs[$seed->module_dir])) {
                    continue;
                }

                $unifiedSearchFields = [];
                $innerJoins = [];
                foreach ($unifiedSearchModules[$moduleName]['fields'] as $field => $def) {
                    $listViewCheckField = strtoupper($field);
                    // Check to see if the field is in listview defs
                    // Check to see if field is in original list view defs (in case we are using custom layout defs)
                    if (
                        empty($listViewDefs[$seed->module_dir][$listViewCheckField]['default']) &&
                        !empty($origListViewDefs[$seed->module_dir][$listViewCheckField]['default'])
                    ) {
                        // If we are here then the layout has been customized, but the field is still needed for query
                        // creation
                        $listViewDefs[$seed->module_dir][$listViewCheckField] = $origListViewDefs[$seed->module_dir][$listViewCheckField];
                    }

                    if (!empty($def['innerjoin'])) {
                        if (empty($def['db_field'])) {
                            continue;
                        }
                        $def['innerjoin'] = str_replace('INNER', 'LEFT', $def['innerjoin']);
                    }

                    if (isset($seed->field_defs[$field]['type'])) {
                        $type = $seed->field_defs[$field]['type'];
                        if ($type === 'int' && !is_numeric($searchQuery)) {
                            continue;
                        }
                    }

                    $unifiedSearchFields[$moduleName][$field] = $def;
                    $unifiedSearchFields[$moduleName][$field]['value'] = $searchQuery;
                }

                /*
                 * Use searchForm2->generateSearchWhere() to create the search query, as it can generate SQL for the full set of comparisons required
                 * generateSearchWhere() expects to find the search conditions for a field in the 'value' parameter of the searchFields entry for that field
                 */
                require_once $beanFiles[$beanName];
                $seed = new $beanName();

                require_once $this->searchFormPath;
                $searchForm = new $this->searchFormClass($seed, $moduleName);

                $searchForm->setup(
                    [$moduleName => []],
                    $unifiedSearchFields,
                    '',
                    'saved_views'
                );
                $whereClauses = $searchForm->generateSearchWhere();
                //add inner joins back into the where clause
                $params = ['custom_select' => ""];
                foreach ($innerJoins as $field => $def) {
                    if (isset($def['db_field'])) {
                        foreach ($def['db_field'] as $dbfield) {
                            $whereClauses[] = $dbfield . " LIKE '" . DBManagerFactory::getInstance()->quote($searchQuery) . "%'";
                        }
                        $params['custom_select'] .= ", $dbfield";
                        $params['distinct'] = true;
                    }
                }

                // Requêtes custom OpenCRM
                switch ($beanName) {
                    case 'OPS_dossier':
                        $whereClauses[] = $this->buildWhereDossier(DBManagerFactory::getInstance()->quote($searchQuery));
                        break;
                    default:
                        break;
                }

                if (!empty($whereClauses)) {
                    $where = '((' . implode(' ) OR ( ', $whereClauses) . '))';
                } else {
                    $where = '';
                }

                $filter_fields = $this->buildFilterFields($seed);

                $listData = $listViewData->getListViewData($seed, $where, 0, -1, $filter_fields, $params);

                $moduleCounts[$moduleName] = $listData['pageData']['offsets']['total'];

                foreach ($listData['data'] as $hit) {
                    $moduleResults[$moduleName][] = $hit['ID'];
                }
            }
        }

        return [
            'hits' => $moduleCounts,
            'modules' => $moduleResults
        ];
    }

    /**
     * Build filter fields for list view data search
     * @param SugarBean $bean
     * @return array
     */
    protected function buildFilterFields(SugarBean $bean): array
    {
        $parsedFilterFields = array();
        $excludedRelationshipFields = array();

        foreach ($bean->field_defs as $fieldName => $fieldDefinition) {
            $type = $fieldDefinition['type'] ?? '';
            $listShow = $fieldDefinition['list-show'] ?? true;

            if ($type === 'link' || $listShow === false) {
                continue;
            }

            $linkType = $fieldDefinition['link_type'] ?? '';
            if ($linkType === 'relationship_info') {
                $excludedRelationshipFields[] = $fieldName;
                $relationshipFields = $fieldDefinition['relationship_fields'] ?? [];
                if (!empty($relationshipFields)) {
                    foreach ($relationshipFields as $relationshipField) {
                        $excludedRelationshipFields[] = $relationshipField;
                    }
                }
            }

            $parsedFilterFields[] = $fieldName;
        }

        return array_diff($parsedFilterFields, $excludedRelationshipFields);
    }

    /**
     * Fonction d'ajout de recherches sur le demandeur, le tiers ou le beneficiaire
     *
     * @param string $search_term
     * @return array
     */
    protected function buildWhereDossier($search_term)
    {
        # Ajout de la recherche sur le demandeur de la demande
        $custom_where[] = 'ops_dossier.id IN (
            SELECT ops_dossier.id
            FROM ops_dossier
            INNER JOIN ops_individu_ops_dossier ON ops_individu_ops_dossier.ops_dossier_id = ops_dossier.id
            INNER JOIN ops_individu ON ops_individu.id = ops_individu_ops_dossier.ops_individu_id
            WHERE (ops_individu.first_name LIKE "' . $search_term . '" OR ops_individu.last_name LIKE "' . $search_term . '")
        )';

        # Ajout de la recherche sur le Tiers de la demande
        $custom_where[] = 'ops_dossier.ops_personne_morale IN (
            SELECT ops_personne_morale.id
            FROM ops_personne_morale 
            WHERE (ops_personne_morale.name LIKE "' . $search_term . '%")
        )';

        # Ajout de la recherche sur le bénéficiaire de la demande
        $custom_where[] = 'ops_dossier.beneficiaire_id IN (
            SELECT ops_individu.id
            FROM ops_individu 
            WHERE (ops_individu.first_name LIKE "' . $search_term . '%" OR ops_individu.last_name LIKE "' . $search_term . '%")
        )';

        return implode(' ) OR ( ', $custom_where);
    }
}