<?php

require_once 'custom/include/libs/tcpdf/tcpdf.php';
require_once 'modules/OPS_generateur_champ/generateur_champ.php';

use ODE\Helper\OdeArrayHelper;
use ODE\Helper\OdeDateHelper;
use ODE\Model\DossierModel;
use ODE\Model\DispositifModel;
use ODE\Model\IndividuModel;
use ODE\Model\PersonneMoraleModel;
use ODE\Model\GenerateurFormulaireModel;
use ODE\Model\GenerateurVueModel;


class OdePDF extends TCPDF
{


    private $config;
    private $header;

    /**
     * Constructeur de la classe OdePDF
     *
     * @access public
     * @name __construct
     * @return void
     */
    public function __construct()
    {
        $this->config = $this->_getConfig();
        $this->header = false;
        parent::__construct(PDF_PAGE_ORIENTATION, 'mm', 'A4', true, 'UTF-8', false);
    }

    /**
     * @access public
     * @name generer()
     * Fonction qui genere le document PDF 
     *
     *  @param string            $dossier_id: L'id du dossier
     *  @return string           $xxx: xxxx 
     */
    public function genererPDF($dossier_id = null, $format = null)
    {

        // $GLOBALS['log']->fatal(" OdePDF :: genererPDF => dossier_id = " . print_r( $dossier_id, true ) );

        do {

            // On vérifie que l'id du dossier n'est pas null ou empty
            if ($dossier_id === null || empty($dossier_id)) {
                $GLOBALS['log']->fatal(" OdePDF :: genererPDF => L'id du dossier à télécharger en pdf est vide ");
                break;
            }

            // On récupére les champs du dossier ( champs du formulaire associé au dispositif )
            $champs_dossier = $this->_getChampsDossier($dossier_id);
            if (!is_array($champs_dossier) || count($champs_dossier) === 0) {
                $GLOBALS['log']->fatal(" OdePDF :: genererPDF => Les champs du formulaire n'ont pas pu etre récupérer ");
                break;
            }

            // On récupére les valeurs des champs 
            $champs = $this->_getChampsValues($champs_dossier, $dossier_id);
            if (!is_array($champs) || count($champs) === 0) {
                $GLOBALS['log']->fatal(" OdePDF :: genererPDF => Les valeurs des champs du dossier n'ont pas pu etre récupérer ");
                break;
            }

            // On récupere les données du header
            $this->header = $this->_getHeaderData($dossier_id);

            // Initialiser les parametres du PDF
            $this->_initParamsPDF();

            // On construit le corps du PDF
            $this->_initCorpsPDF($champs);

            // On sort le PDF
            if ($format === null) {
                $this->Output('dossier_' . $this->header['dossier_num'] . '.pdf', 'D');
            } else {
                $this->Output('dossier_' . $this->header['dossier_num'] . '.pdf', 'F');
            }
        } while (0);
    }

    /**
     * @access private
     * @name _initCorpsPDF()
     * Fonction qui construit le corps du pDF
     *
     *  @param string            $champs_pdf: Les champs du dossier
     */
    private function _initCorpsPDF($champs_pdf)
    {

        $corps_html = '';

        if ($this->config['in_colonnes'] == "1") {

            $champs_pdf_formated = array();
            foreach ($champs_pdf as $libelle_onglet => $champs) {
                $champs_pdf_formated[$libelle_onglet] = array_chunk($champs, 2, true);
            }

            foreach ($champs_pdf_formated as $libelle_onglet => $champs) {

                $corps_html .= '<span style="' . $this->config['onglet'] . '"><br>' . $libelle_onglet . ' </span><br />';
                $corps_html .= '<br />';
                $corps_html .= '<table cellspacing="4" cellpadding="3" >';

                foreach ($champs as $champ) {
                    $corps_html .= $this->_getTrChamp($champ);
                }
                $corps_html .= '</table>';
            }
        } else {

            foreach ($champs_pdf as $libelle_onglet => $champs) {
                $corps_html .= '<span style="' . $this->config['onglet'] . '"><br>' . $libelle_onglet . ' </span><br />';
                $corps_html .= '<br />';
                $corps_html .= '<table cellspacing="3" cellpadding="3" >';
                foreach ($champs as $champ_label => $champ_value) {
                    $corps_html .= '<tr>';
                    $corps_html .= '<td><span style="' . $this->config['label'] . '">' . $champ_label . ' : </span></td>';
                    $corps_html .= '<td bgcolor="#c3e7fd"><span style="' . $this->config['champ'] . '">' . nl2br(html_entity_decode($champ_value)) . '</span></td>';
                    $corps_html .= '</tr>';
                }
                $corps_html .= '</table>';
            }
        }

        $this->writeHTML($corps_html, true, false, true, false, '');
    }

    /**
     * @access private
     * @name _getTrChamp()
     * Fonction qui retourne les champs en format TR>TD
     *
     *  @param array              $champ: xxxx
     *  @return string            $champ_html: xxxx 
     */
    private function _getTrChamp($champ)
    {
        $champ_html = '';

        if (is_array($champ) && count($champ) > 0 && count($champ) < 3) {

            switch (count($champ)) {
                case 1:
                    $champ[] = "";
                    break;
                case 2:
                    break;
            }

            if (count($champ) === 2) {
                $champ_html .= '<tr>';
                foreach ($champ as $champ_label => $champ_value) {
                    $champ_label_display = ($champ_label != "0") ? $champ_label . " : " : "";
                    $td_color = ($champ_label != "0") ? 'bgcolor="#c3e7fd"' : '';
                    $champ_html .= '<td><span style="' . $this->config['label'] . ' ">' . $champ_label_display . '</span></td>';
                    $champ_html .= '<td ' . $td_color . '><span style="' . $this->config['champ'] . '">' . nl2br(html_entity_decode($champ_value))  . '</span></td>';
                }
                $champ_html .= '</tr>';
            }
        }

        return $champ_html;
    }

    /**
     * @access private
     * @name _initParamsPDF()
     * Fonction qui initialise les parametres du PDF
     * 
     */
    private function _initParamsPDF()
    {
        $margin_header = 5;
        $margin_footer = 10;
        $interligne = 1.05;
        $titre =  mb_strtoupper($this->header['dossier_name'], 'UTF-8');

        // set document information
        $this->SetCreator($this->config["createur"]);
        $this->SetAuthor($this->config["createur"]);
        $this->SetTitle(str_replace("&#039;", "'", $titre));

        // set header and footer fonts
        $this->setHeaderFont(array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));
        $this->setFooterFont(array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
        // set default monospaced font
        $this->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);

        //set margins
        $this->SetMargins($this->config["margin"]["left"], $this->config["margin"]["top"], $this->config["margin"]["right"]);
        $this->SetHeaderMargin($margin_header);
        $this->SetFooterMargin($margin_footer);

        //set auto page breaks
        $this->SetAutoPageBreak(TRUE, $this->config["margin"]["bottom"]);

        //set image scale factor
        $this->setImageScale(PDF_IMAGE_SCALE_RATIO);
        //set some language-dependent strings
        //$pdf->setLanguageArray($l);
        // set font
        $this->SetFont($this->config["police_pdf"], '', 11);

        //définiton de l'interligne
        $this->setCellHeightRatio($interligne);
        //Ajout d'une page
        $this->AddPage();
    }

    /**
     * @access public
     * @name Header()
     * Fonction qui construit le header du pdf  ( Surcharge fonction TCPDF::Header )
     *
     */
    public function Header()
    {

        $table =    '<table>

                        <tr>
                            <td colspan="2"><img height="60px" width="auto" src="' . $this->config["logo_chemin"] . '" align="left"></td>
                            <td ></td>
                            <td ></td>
                            <td ></td>
                            <td colspan="2"><span style="font-size:12px;text-align:right;">' . $this->header["dossier_date"] . ' </span> </td>
                        </tr>

                        <tr>
                            <td ></td>
                            <td colspan="5" style="text-align: center;padding-bottom:10px;"> 
                                <span style="' . $this->config["titre"] . '" >' .  $this->header["dispositif_name"] . '</span> <br />
                                <span style="' . $this->config["titre"] . '" > 
                                    Dossier N° ' . $this->header["dossier_num"] . '<br />
                                </span> 
                                <span style="' . $this->config["titre"] . '" >' .  $this->header["demandeur_name"] . '</span> 
                            </td>
                            <td ></td>
                        </tr>

                    </table>';


        //Ecriture du tableau dans l'êntete
        $this->writeHTML($table, true, false, true, false, '');
    }

    // Page footer
    public function Footer()
    {
        // Position at 15 mm from bottom
        $this->SetY(-15);
        // Set font
        $this->SetFont('helvetica', 'I', 8);
        // Page number
        $this->Cell(0, 10, 'Page ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, false, 'C', 0, '', 0, false, 'T', 'M');
    }

    /**
     * @access private
     * @name _getChampsDossier()
     * Fonction qui retourne les champs du dossier ( champs du formulaire associé au dispositif )
     *
     *  @param string            $xxx: xxxx
     *  @return array            $xxx: xxxx 
     */
    private function _getChaumpsDossier()
    {
    }

    /**
     * @access private
     * @name _getChampsDossier()
     * Fonction qui retourne les champs du dossier ( champs du formulaire associé au dispositif )
     *
     *  @param string            $dossier_id: L'id du dossier 
     *  @return array            $header_data: Les données d'initilisation du Header 
     */
    private function _getHeaderData($dossier_id)
    {

        $header_data = array(
            "dossier_num" => "",
            "dossier_name" => "",
            "dossier_date" => "Transmis le : ",
            "dispositif_name" => "",
            "demandeur_name" => "",
        );

        $dossierModel = new DossierModel($dossier_id);
        $dossier = $dossierModel->getBean();
        if (!is_array($dossier) || count($dossier) === 0) {
            $GLOBALS['log']->fatal(" OdePDF :: _getHeaderData => Le dossier n'a pas pu etre récupéré ! ");
            return $header_data;
        }

        if (!empty($dossier['num_dossier'])) {
            $header_data["dossier_num"] = $dossier['num_dossier'];
        }

        if (!empty($dossier['name'])) {
            $header_data["dossier_name"] = $dossier['name'];
        }

        if (!empty($dossier['date_entered'])) {
            $header_data["dossier_date"] = "Transmis le : " . OdeDateHelper::toHTML($dossier['date_entered']);
        }

        $demandeur_id = $dossierModel->getDemandeurId();
        $demandeur_type = $dossierModel->getDemandeurType();
        if (!empty($demandeur_id) && !empty($demandeur_type)) {
            if ($demandeur_type === "Personne Morale") {
                $personneMoraleModel = new PersonneMoraleModel($demandeur_id);
                $demandeur = $personneMoraleModel->getBean();
                if (is_array($demandeur) && count($demandeur) > 0) {
                    $header_data["demandeur_name"] = $demandeur["name"];
                }
            }
            if ($demandeur_type === "Individu") {
                $individuModel = new IndividuModel($demandeur_id);
                $demandeur = $individuModel->getBean();
                if (is_array($demandeur) && count($demandeur) > 0) {
                    $header_data["demandeur_name"] =  $demandeur["salutation"] . " " . $demandeur["first_name"] . " " . $demandeur["last_name"];
                }
            }
        }

        $dispositif_id = $dossierModel->getDispositifId();
        if (!empty($dispositif_id)) {
            $dispositifModel = new DispositifModel($dispositif_id);
            $dispositif = $dispositifModel->getBean();
            if (is_array($dispositif) && count($dispositif) > 0) {
                $header_data["dispositif_name"] = $dispositif["name"];
            }
        }

        return $header_data;
    }

    /**
     * @access private
     * @name _getChampsValues()
     * Fonction qui retourne les champs du dossier avec leurs valeurs
     *
     *  @param array              $champs_dossier: Champs du formulaire associé au dispositif 
     *  @param string             $dossier_id: L'id du dossier 
     *  @return array             $champs_pdf: xxxx 
     */
    private function _getChampsValues($champs_dossier, $dossier_id)
    {
        $champs_pdf = array();
        if (count($champs_dossier) > 0 && !empty($dossier_id)) {
            $GenerateurChamp = new GenerateurChamp($dossier_id, 'pdf');
            foreach ($champs_dossier as $onglet_libelle => $champs) {
                foreach ($champs as $champ) {
                    $champs_pdf[$onglet_libelle][$champ['libelle']] = $GenerateurChamp->getChampValueDisplay($champ['name']);
                }
            }
        }
        return $champs_pdf;
    }

    /**
     * @access private
     * @name _getChampsDossier()
     * Fonction qui retourne les champs du dossier ( champs du formulaire associé au dispositif )
     *
     *  @param string            $dossier_id: L'id du dossier 
     *  @return array            $champs_dossier: champs du formulaire associé au dispositif 
     */
    private function _getChampsDossier($dossier_id)
    {
        $champs_dossier = [];

        do {

            $DossierModel = new DossierModel($dossier_id);
            $dispositif_id = $DossierModel->getDispositifId();
            if (empty($dispositif_id)) {
                $GLOBALS['log']->fatal(" OdePDF :: _getChampsDossier => Le dispositif n'a pas pu etre récupéré. ");
                break;
            }

            $DispositifModel = new DispositifModel($dispositif_id);
            $dispositif = $DispositifModel->getBean();
            if (!is_array($dispositif) || count($dispositif) === 0) {
                $GLOBALS['log']->fatal(" OdePDF :: _getChampsDossier => Le dispositif id = " . $dispositif_id . " est introuvable");
                break;
            }

            // On récupere la vue agent du formulaire associé au dispositif
            $formulaire_id = $DispositifModel->getFormulaireId();
            $GenerateurFormulaireModel = new GenerateurFormulaireModel($formulaire_id);
            $agent_vue_id = $GenerateurFormulaireModel->getVueAgentId();
            if (empty($agent_vue_id)) {
                $GLOBALS['log']->fatal(" OdePDF :: _getChampsDossier => La vue agent associé formulaire id = " . $formulaire_id . " est introuvable");
                break;
            }

            $vueModel = new GenerateurVueModel($agent_vue_id);
            $vue = $vueModel->getBean();
            if (!is_array($vue) || count($vue) === 0) {
                $GLOBALS['log']->fatal(" OdePDF :: _getChampsDossier => La id = " . $agent_vue_id . " est introuvable");
                break;
            }

            $onglets = $vueModel->getOnglets();
            foreach ($onglets as $onglet) {
                if (!empty($onglet['libelle']) && !empty($onglet['champs']) && is_array($onglet['champs']) && count($onglet['champs']) > 0) {
                    foreach ($onglet['champs'] as $ligne) {
                        foreach ($ligne as $champ) {
                            if ($champ['type'] !== "vide") {
                                $champs_dossier[$onglet['libelle']][] = $champ;
                            }
                        }
                    }
                }
            }
        } while (0);

        return $champs_dossier;
    }

    /**
     * @access private
     * @name _getConfig()
     * Fonction qui retourne la configuration du pdf dynamique
     *
     *  @return array          $config: la configuration du pdf dynamique 
     */
    private function _getConfig()
    {
        global $sugar_config;

        $config = array(
            "createur" => "OpenSub",
            "logo_chemin" => "themes/default/images/pdf_logo.png",
            "police_pdf" => "arial",
            "in_colonnes" => false,
            "titre" => "",
            "onglet" => "",
            "label" => "",
            "champ" => "",
            "margin" => array("top" => 2, "bottom" => 2, "right" => 20, "left" => 20),

        );

        // On récupere les données de configuration
        if (!empty($sugar_config['ops_pdf_dynamique']['general']['margin']['top']) && intval($sugar_config['ops_pdf_dynamique']['general']['margin']['top']) > 1) {
            $config["margin"]["top"] = intval($sugar_config['ops_pdf_dynamique']['general']['margin']['top']);
        }

        if (!empty($sugar_config['ops_pdf_dynamique']['general']['margin']['bottom']) && intval($sugar_config['ops_pdf_dynamique']['general']['margin']['bottom']) > 1) {
            $config["margin"]["bottom"] = intval($sugar_config['ops_pdf_dynamique']['general']['margin']['bottom']);
        }

        if (!empty($sugar_config['ops_pdf_dynamique']['general']['margin']['right']) && intval($sugar_config['ops_pdf_dynamique']['general']['margin']['right']) > 20) {
            $config["margin"]["right"] = intval($sugar_config['ops_pdf_dynamique']['general']['margin']['right']);
        }

        if (!empty($sugar_config['ops_pdf_dynamique']['general']['margin']['left']) && intval($sugar_config['ops_pdf_dynamique']['general']['margin']['left']) > 20) {
            $config["margin"]["left"] = intval($sugar_config['ops_pdf_dynamique']['general']['margin']['left']);
        }

        if (!empty($sugar_config['ops_pdf_dynamique']['general']['createur'])) {
            $config["createur"] = $sugar_config['ops_pdf_dynamique']['general']['createur'];
        }

        if (!empty($sugar_config['ops_pdf_dynamique']['general']['logo_chemin'])) {
            $config["logo_chemin"] = $sugar_config['ops_pdf_dynamique']['general']['logo_chemin'];
        }

        if (!empty($sugar_config['ops_pdf_dynamique']['general']['police_pdf'])) {
            $config["police_pdf"] = $sugar_config['ops_pdf_dynamique']['general']['police_pdf'];
        }

        if (!empty($sugar_config['ops_pdf_dynamique']['general']['in_colonnes']) && $sugar_config['ops_pdf_dynamique']['general']['in_colonnes'] == '1') {
            $config["in_colonnes"] = true;
        }

        $parametres = ["titre", "onglet", "label", "champ"];
        foreach ($parametres as $param_title) {
            $config[$param_title] = 'font-family:' . $config["police_pdf"] . ';';
            foreach ($sugar_config['ops_pdf_dynamique'][$param_title] as $parametre => $valeur) {
                switch ($parametre) {
                    case 'couleur_pdf':
                        $config[$param_title] .= 'color:' . $valeur . ';';
                        break;
                    case 'taille_pdf':
                        $config[$param_title] .= 'font-size:' . $valeur . 'px;';
                        break;
                    case 'pdf_gras':
                        if ($valeur == 1) {
                            $config[$param_title] .= 'font-weight: bold;';
                        }
                        break;
                    case 'pdf_italique':
                        if ($valeur == 1) {
                            $config[$param_title] .= 'font-style: italic;';
                        }
                        break;
                    case 'pdf_souligner':
                        if ($valeur == 1) {
                            $config[$param_title] .= 'text-decoration: underline;';
                        }
                        break;
                }
            }
        }

        return $config;
    }
}
