<?php

namespace ODE\Mailer;

use BeanFactory;
use ODE\Helper\OdePdfHelper;
use ODE\Helper\ReplaceVariable;

/**
 * Class OdeEmail
 * La classe prend en parametre un tableau de données, certains champs sont obligatoires :
 *       replace (bool): Par défaut la classe remplacera les variables dans l'email ( titre, corps ..), (Valeur par défaut : true)
 *       bean_source_id* (string): L'id du module source pour le remplacement des variables ( Facultatif si replace === false )
 *       bean_source_name* (string): Le nom du module source pour le remplacement des variables ( Facultatif si replace === false )
 *       bean_historisation_id (string): L'id du module pour historiser le mail envoyé ( Obligatoire si on souhaite historiser l'email)
 *       bean_historisation_name (string): Le nom du module pour historiser le mail envoyé ( Obligatoire si on souhaite historiser l'email)
 *       email_template_id* (string): l'id du template à utiliser, module 'EmailTemplate'
 *       dest_to* (string): Adresses mails séparé par un point virgule ( Obligatoire si dest_cc et dest_cci sont vides)
 *       dest_cc* (string): Adresses mails séparé par un point virgule ( Obligatoire si dest_to et dest_cci sont vides)
 *       dest_cci* (string): Adresses mails séparé par un point virgule ( Obligatoire si dest_to et dest_cc sont vides)
 *       genere_dossier_pdf(bool): Par défaut à false, passez à true si vous souhaitez ajouter le recap du dossier pdf dans les pieces jointes du mail
 * 
 * @package Ode\Mailer
 */
class OdeEmail
{

    /** @var bool */
    public $statut = true;

    /** @var bool */
    public $log = false;

    /** @var bool */
    public $replace = true;

    /** @var string */
    public $bean_source_id = '';

    /** @var string */
    public $bean_source_name = '';

    /** @var string */
    public $bean_historisation_id = '';

    /** @var string */
    public $bean_historisation_name = '';

    /** @var string */
    public $email_template_id = '';

    /** @var string */
    public $subject = '';

    /** @var string */
    public $body_html = '';

    /** @var string */
    public $body = '';

    /** @var array */
    public $dest_to = [];

    /** @var array */
    public $dest_cc = [];

    /** @var array */
    public $dest_cci = [];

    /** @var bool */
    public $genere_dossier_pdf = false;

    /** @var bool */
    public $ajouter_justificatifs = false;

    /** @var string */
    public $type_justificatifs = '';

    /**  @var array */
    public $attachments = [];

    /**
     * OdeEmail constructor.
     *       
     * @param array $data
     */
    public function __construct($data = [])
    {
        // On vérifie qu'on un bien un tableau non vide
        if (!is_array($data) || count($data) === 0) {
            $GLOBALS['log']->fatal("OdeEmail::__construct => la classe doit etre initialiser avec un tableau de valeurs");
            $this->statut = false;
            return;
        }

        // On vérifie qu'on a bien récupéré le bean source si l'option replace est à true ( par défaut )
        $this->setBeanSource($data);

        // On vérifie qu'on bien récupéré le bean qui servira pour l'historisation, dans tout les cas l'email sera envoyé
        $this->setBeanHistorisation($data);

        // On vérifie que le template existe bien, on initialise les différents textes qui composent le mail
        $this->setEmailTemplate($data);

        // On initialise les destinataires dest_to , dest_cc et dest_cci
        $this->setDestinataires($data);

        // On initialise les pieces jointes à envoyer
        $this->setPiecesJointes($data);

        // On remplace les variables dans les différents textes
        $this->replaceVariables();
    }

    /**
     * Fonction qui vérifie et qui initialise les 3 variables de classes 'replace', 'bean_source_id' et 'bean_source_name' utilisées pour le remplacement des variables
     * @param array    $data
     * 
     * @return void
     */
    protected function setBeanSource($data = [])
    {
        $this->replace = !isset($data['replace']) || $data['replace'] !== false;

        if ($this->replace) {

            $this->bean_source_id = !empty($data['bean_source_id']) ? $data['bean_source_id'] : '';
            $this->bean_source_name = !empty($data['bean_source_name']) ? $data['bean_source_name'] : '';

            if (empty($this->bean_source_id)) {
                $GLOBALS['log']->fatal("OdeEmail::setBeanSource => le parametre 'bean_source_id' est vide ou manquant");
            }

            if (empty($this->bean_source_name)) {
                $GLOBALS['log']->fatal("OdeEmail::setBeanSource => le parametre 'bean_source_name' est vide ou manquant");
                $this->statut = false;
                return;
            }

            $obj_bean = BeanFactory::getBean($this->bean_source_name, $this->bean_source_id);
            if (empty($obj_bean->id)) {
                $GLOBALS['log']->fatal("OdeEmail::setBeanSource => l'enregistrement du module " . $this->bean_source_name . " est introuvable");
                $this->statut = false;
            }
        }
    }

    /**
     * Fonction qui vérifie les deux variables 'bean_historisation_id' et 'bean_historisation_name' et qui initialise la variable 'log'
     * Pour désactiver l'historisation, il suffit de ne pas passer un des deux parametres
     * @param array    $data
     * 
     * @return void
     */
    protected function setBeanHistorisation($data = [])
    {

        $this->bean_historisation_id = !empty($data['bean_historisation_id']) ? $data['bean_historisation_id'] : '';
        $this->bean_historisation_name = !empty($data['bean_historisation_name']) ? $data['bean_historisation_name'] : '';
        if (empty($this->bean_historisation_id) || empty($this->bean_historisation_name)) return;

        $obj_bean = BeanFactory::getBean($this->bean_historisation_name, $this->bean_historisation_id);
        if (!empty($obj_bean->id)) {
            $this->log = true;
        } else {
            $GLOBALS['log']->fatal("OdeEmail::setBeanHistorisation => l'enregistrement du module " . $this->bean_historisation_name . " est introuvable, l'mail ne sera pas historiser");
        }
    }

    /**
     * Fonction qui vérifie les trois variables 'body', 'body_html' et 'subject' à partir du module EmailTemplate
     * @param array    $data
     * 
     * @return void
     */
    protected function setEmailTemplate($data = [])
    {
        $this->email_template_id = !empty($data['email_template_id']) ? $data['email_template_id'] : '';
        $obj_email_template = BeanFactory::getBean('EmailTemplates', $this->email_template_id);
        if (empty($obj_email_template->id)) {
            $GLOBALS['log']->fatal("OdeEmail::setEmailTemplate => le modele de mail ( EmailTemplate , " . $this->email_template_id . " ) est introuvable");
            $this->statut = false;
            return;
        }

        $this->body         = $obj_email_template->body;
        $this->body_html    = $obj_email_template->body_html;
        $this->subject      = $obj_email_template->subject;
    }

    /**
     * Fonction qui vérifie les trois variables 'dest_to', 'dest_cc' et 'dest_cci' et les converti en tableau
     * @param array    $data
     * 
     * @return void
     */
    protected function setDestinataires($data = [])
    {
        $this->dest_to = !empty($data['dest_to']) ? $this->getEmailsFromString($data['dest_to']) : [];
        $this->dest_cc = !empty($data['dest_cc']) ? $this->getEmailsFromString($data['dest_cc']) : [];
        $this->dest_cci = !empty($data['dest_cci']) ? $this->getEmailsFromString($data['dest_cci']) : [];

        if (count($this->dest_to) === 0 && count($this->dest_cc) === 0 && count($this->dest_cci) === 0) {
            $GLOBALS['log']->fatal("OdeEmail::setDestinataires => Il faut au moins que un des champs du tableau ['dest_to','dest_cc','dest_cci'] ne soit pas vide");
            $this->statut = false;
            return;
        }
    }

    /**
     * Fonction qui converti des adresses mails au format string en tableau, elle vérifie aussi la validité de l'adresse avant de l'ajouter 
     *  Exemple :
     *      $data['dest_to'] = 'agent1@test.com;agent2@test.com;fausseAdresse.com'    =>   $this->dest_to = [ 'agent1@test.com', 'agent2@test.com' ]
     * 
     * @param string  $emails_string
     * 
     * @return array  $emails
     */
    protected function getEmailsFromString($emails_string = '')
    {

        if (empty($emails_string) || !is_string($emails_string)) return [];

        $emails = [];
        $emails_array = explode(';', $emails_string);
        foreach ($emails_array as $address_mail) {
            if (filter_var(trim($address_mail), FILTER_VALIDATE_EMAIL)) {
                $emails[] = trim($address_mail);
            } else {
                $GLOBALS['log']->fatal("OdeEmail::getEmailsFromString => l'adresse mail ( " . $address_mail . " ) n'est pas reconnu comme valide");
            }
        }

        return $emails;
    }

    /**
     * Fonction qui initialise les pieces jointes à ajouter au mail
     * @param array  $data
     * 
     * @return void 
     */
    protected function setPiecesJointes($data = [])
    {

        // Fonctionnalité Ajout du dossier au format pdf aux pieces jointes
        $this->genere_dossier_pdf = isset($data['genere_dossier_pdf']) && $data['genere_dossier_pdf'] === true;
        $this->ajouter_justificatifs = isset($data['ajouter_justificatifs']) && $data['ajouter_justificatifs'] === true;
        $this->type_justificatifs = (!empty($data['type_justificatifs']))?$data['type_justificatifs']:'';
        if ($this->statut && $this->genere_dossier_pdf) $this->addDossierPdf();
        if ($this->statut && $this->ajouter_justificatifs && $this->type_justificatifs) $this->addJustificatifs();
    }

    /**
     * Fonction qui génere le récap du dossier au format pdf et l'ajoute aux pieces jointes de la classe
     * 
     * @return void 
     */
    protected function addDossierPdf()
    {
        // Cette fonctionnalité est pour le moment possible uniquement à partir de l'objet OPS_dossier comme module source
        if ($this->bean_source_name !== 'OPS_dossier') {
            $GLOBALS['log']->fatal("OdeEmail::addDossierPdf => Le module source doit obligatoire etre 'OPS_dossier' pour ajouter le récap pdf au mail' ");
            return;
        }

        // On génere le PDF 
        $OdePdf  = new OdePdfHelper($this->bean_source_id, 'usager', 'mail');
        $pdf = $OdePdf->generer();

        // On vérifie que le mail a bien été généré
        if ($pdf['statut'] !== true || empty($pdf['data']['name'])) {
            $GLOBALS['log']->fatal("OdeEmail::addDossierPdf => Erreur lors de la génération de dossier en pdf, il ne sera pas ajouter au mail ");
            return;
        }

        // On ajoute le fichier au tableau de pieces jointes
        $location = (empty($_SERVER['DOCUMENT_ROOT']) || $_SERVER['DOCUMENT_ROOT'] == '/')?getcwd():$_SERVER['DOCUMENT_ROOT'];
        $this->attachments[] = (object)[
            'file_name' => $pdf['data']['name'],
            'file_path' => $location . '/upload/' . $pdf['data']['name']
        ];
    }

    /**
     * Fonction qui ajoute les justificatifs aux pieces jointes de la classe
     * 
     * @return void 
     */
    protected function addJustificatifs(){

        global $sugar_config;
        
        if ($this->bean_source_name !== 'OPS_dossier') {
            $GLOBALS['log']->fatal("OdeEmail::addJustificatifs => Le module source doit obligatoire etre 'OPS_dossier' pour ajouter les justificatifs au mail' ");
            return;
        }

        $objDossier = BeanFactory::getBean('OPS_dossier',$this->bean_source_id);
        $objDossier->load_relationship('ops_dossier_ops_justificatif');
        $justificatifs = $objDossier->ops_dossier_ops_justificatif->get();
        foreach ($justificatifs as $justificatif) {
            $objJustificatif = BeanFactory::getBean('OPS_justificatif',$justificatif);

            if($this->type_justificatifs == 'justificatif_valide' && $objJustificatif->statut != 'valide'){
                continue;
            }
            // On ajoute le fichier au tableau de pieces jointes
            $location = (empty($_SERVER['DOCUMENT_ROOT']) || $_SERVER['DOCUMENT_ROOT'] == '/')?getcwd():$_SERVER['DOCUMENT_ROOT'];
            $location .= "/".$sugar_config['ops_justificatif']['chemin']."/".$objDossier->id;
            $this->attachments[] = (object)[
                'file_name' => $objJustificatif->document_name,
                'file_path' => $location . '/' . $objJustificatif->id
            ];
        }
    }

    /**
     * Fonction qui remplace les variables dans les differents textes du mail, il utilise la classe ODE\Helper\ReplaceVariable
     * 
     * @return void 
     */
    protected function replaceVariables()
    {
        if (!$this->statut || !$this->replace) return;
        $obj_bean = BeanFactory::getBean($this->bean_source_name, $this->bean_source_id);
        $ReplaceVariable = new ReplaceVariable($obj_bean);
        $this->subject = $ReplaceVariable->replace($this->subject);
        $this->body_html = $ReplaceVariable->replace($this->body_html);
        $this->body = $ReplaceVariable->replace($this->body);
    }
}
