<?php

use ODE\Helper\OdeArrayHelper;
use ODE\Helper\OdeStringHelper;
use ODE\Helper\OdeDateHelper;
use ODE\Model\DossierModel;
use ODE\Model\DispositifModel;
use ODE\Model\ReunionModel;
use ODE\Mailer\OdeEmail;
use ODE\Mailer\OdeMailer;
use TCPDF;
use ODE\Generateur\Factory\OdeFieldFactory;

class OPS_dossier extends Basic
{


    public $new_schema = true;
    public $module_dir = 'OPS_dossier';
    public $object_name = 'OPS_dossier';
    public $table_name = 'ops_dossier';
    public $importable = true;

    public $id;
    public $name;
    public $date_entered;
    public $date_modified;
    public $modified_user_id;
    public $modified_by_name;
    public $created_by;
    public $created_by_name;
    public $description;
    public $deleted;
    public $created_by_link;
    public $modified_user_link;
    public $assigned_user_id;
    public $assigned_user_name;
    public $assigned_user_link;
    public $SecurityGroups;
    public $num_dossier;
    public $type_tiers;
    public $canal;
    public $brouillon;
    public $doublon;

    public function bean_implements($interface)
    {
        switch ($interface) {
            case 'ACL':
                return true;
        }

        return false;
    }


    /*******************************************************************************************************************************************/
    /************************************************************* Création Dossier ************************************************************/
    /*******************************************************************************************************************************************/

    /**
     * @access public
     * @name createDossier()
     * Fonction pour créer un dossier
     *
     *  @param array                           $formulaire : Tableau des champs du formulaire
     *  @return string|boolean                 $dossier_id : retourne l'id du dossier crée sinon false
     */
    public static function createDossier($formulaire)
    {

        // Vérification des champs indispensable pour la création d'un dossier sur OpenSub
        if (OPS_dossier::isFormulaireValid($formulaire) === false) return false;

        // On récupere les champs de la vue associé au dispositif
        $DispositifModel = new DispositifModel($formulaire['dispositif']);
        $champs_vue = $DispositifModel->getChampsVueAgent();

        // Si les champs sont vide, on retourne false
        if (!is_array($champs_vue) || count($champs_vue) === 0) {
            $GLOBALS['log']->fatal(" OPS_dossier :: createDossier => Les champs de la vue agent associé au dispositif n'ont pas pu etre récupéré. ");
            return false;
        }

        $formulaire["brouillon"] = 'non';
        $formulaire["ordre_onglet"] = 0;

        if (!isset($formulaire["canal"]) || empty($formulaire["canal"])) {

            $formulaire["canal"] = 'crm';
        }

        // On initialise les champs à créer ( Divise les champs en deux tableau basic / custom )
        $dossier_champs = OPS_dossier::initDataDossier($formulaire, $champs_vue, false);
        if (!is_array($dossier_champs) || count($dossier_champs) === 0) {
            $GLOBALS['log']->fatal(" OPS_dossier :: createDossier => Echec de l'initialisation des champs dossier => dossier non crée . ");
            return false;
        }
        $dossier_champs['basic']['beneficiaire_id'] = $formulaire['beneficiaire_id'];
        $dossier_champs['basic']['ops_personne_morale'] = $formulaire['ops_personne_morale'];

        $dossier_champs['extended']['dispositif'] = $formulaire['dispositif'];
        $dossier_champs['extended']['demandeur'] = $formulaire['demandeur'];

        // On crée le dossier sur la base de champs effectif préremplis (fournis via formulaire vs champs disponible|possibles de la vue assiciée)
        $dossier_id = OPS_dossier::createBeanDossier($dossier_champs);

        return (!empty($dossier_id)) ? $dossier_id : false;
    }

    public static function upDateDossierByEtape($dossier_id, $formulaire, $etape, $brouillon = '')
    {

        global $sugar_config;

        $obj_dossier = BeanFactory::getBean('OPS_dossier', $dossier_id);
        if (empty($obj_dossier->id)) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Le dossier id = " . $dossier_id . " n'a pas pu etre récupéré. ");
            return false;
        }

        if (empty($obj_dossier->type_tiers)) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Le dossier id = " . $obj_dossier->id . " ne dispose pas d'un type tiers. ");
            return false;
        } else {
            $formulaire["type_tiers"] = $obj_dossier->type_tiers;
        }

        // Vérification des champs indispensable pour la création d'un dossier sur OpenSub
        if (OPS_dossier::isFormulaireValid($formulaire) === false) {
            return false;
        }

        // On récupere les champs de la vue associé au dispositif
        $DispositifModel = new DispositifModel($formulaire['dispositif']);

        // Dans le cas du partenaire

        if (array_key_exists('vue', $formulaire) && $formulaire['vue'] == 'partenaire') {
            $champs_vue = $DispositifModel->getChampsVuePartenaireByEtape($etape);
            $etapes = $DispositifModel->getEtapesVuePartenaire($etape);
        } else {
            $champs_vue = $DispositifModel->getChampsVueUsagerByEtape($etape);
            $etapes = $DispositifModel->getEtapesVueUsager($etape);
        }

        if (is_array($etapes) && count($etapes) == $etape && (!isset($formulaire["canal"]) || isset($formulaire["terminer"]))) {
            if ($brouillon == "brouillon") {
                $formulaire["brouillon"] = 'oui';
            } else {
                $formulaire["brouillon"] = 'non';
            }
            $formulaire["ordre_onglet"] = 0;
        } else {
            $formulaire["ordre_onglet"] = $etape;
        }

        if (isset($formulaire["terminer"])) {
            if ($formulaire["terminer"] == "true" && $obj_dossier->brouillon == "oui") {
                $formulaire["brouillon"] = 'non';
                $formulaire["creation"] = true;
            }
            $termine = $formulaire["terminer"];
            unset($formulaire["terminer"]);
        }


        // Si les champs sont vide, on retourne false
        if (!is_array($champs_vue) || count($champs_vue) === 0) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Les champs de la vue agent associé au dispositif n'ont pas pu etre récupéré. ");
            return false;
        }

        // On initialise les champs à créer ( Divise les champs en deux tableau basic / custom )
        $dossier_champs = OPS_dossier::getDataDossier($formulaire, $champs_vue);
        if (!is_array($dossier_champs) || count($dossier_champs) === 0) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Echec de l'initialisation des champs dossier => dossier non crée . ");
            return false;
        }

        unset($dossier_champs["basic"]["type_tiers"]);

        // On crée le dossier sur la base de champs effectif préremplis (fournis via formulaire vs champs disponible|possibles de la vue assiciée)
        $dossier_id = OPS_dossier::editBeanDossier($obj_dossier, $dossier_champs, $formulaire['brl']);

        if (isset($termine) && $termine == "true" && !empty($dossier_id) && $obj_dossier->incomplet == 0 && $formulaire["creation"]) {
            // Initialisation du statut et de l'étape du dossier
            $obj_dossier = OPS_dossier::initStatutEtapeDossier($obj_dossier, $formulaire['dispositif']);
            $dossier_id = $obj_dossier->save();

            if ($obj_dossier->canal == 'hors-compte') {
                $template = $sugar_config['opensocle']['notif_usager_dossier_hc_creation'];
            } else {
                $template = $sugar_config['opensocle']['notif_usager_dossier_creation'];
            }
            if (isset($template) && !empty($template)) {
                $obj_individu = new OPS_individu();
                $obj_individu->retrieve($obj_dossier->ops_individu_id);

                # On initialise le mailer 
                $mailer = new OdeMailer();

                # On initialise l'email 
                $ode_email = new OdeEmail([
                    'bean_source_id' => $obj_dossier->id,
                    'bean_source_name' => 'OPS_dossier',
                    'bean_historisation_id' => $obj_dossier->ops_individu_id,
                    'bean_historisation_name' => "OPS_individu",
                    'email_template_id' => $template,
                    'dest_to' => $obj_individu->email1,
                ]);

                # On déclenche l'envoie
                $mailer->send($ode_email);

            }
            else{
                $GLOBALS['log']->fatal("DossiersService - sendNotification : Pas te template mail associé à la complétude d'un dossier");
            }
        }

        return (!empty($dossier_id)) ? $dossier_id : false;
    }

    /**
     * @access public
     * @name createDossier()
     * Fonction pour créer un dossier
     *
     *  @param array                           $formulaire : Tableau des champs du formulaire
     *  @return string|boolean                 $dossier_id : retourne l'id du dossier crée sinon false
     */
    public static function createDossierByEtape($donnees)
    {
        $canal = 'internet';

        if (isset($donnees["dispositif"]["canal"]) && !empty($donnees["dispositif"]["canal"])) {

            $canal = $donnees["dispositif"]["canal"];
        }

        $obj_dossier = BeanFactory::newBean("OPS_dossier");
        $obj_dossier->type_tiers = 'OPS_individu';
        $obj_dossier->brouillon = "oui";
        $obj_dossier->doublon = "non";
        $obj_dossier->ordre_onglet = 0;
        $obj_dossier->canal = $canal;
        $obj_dossier->ops_individu_id = $donnees["demandeur"]["id"];

        $obj_dossier->ops_personne_morale = $donnees['ops_personne_morale'];
        $obj_dossier->beneficiaire_id = $donnees['beneficiaire_id'];

        // On paramètre toujours en TEMP tant que la demande n'est pas finalisée ou le brouillon enregistré
        if ($obj_dossier->canal != 'hors-compte') {
        $obj_dossier->name = 'TEMP';
        } else {
            $obj_dossier->name = DossierModel::getNumDossier($obj_dossier->id);
        }
        $dossier_id = $obj_dossier->save();

        $obj_dispositif = BeanFactory::getBean("OPS_dispositif", $donnees["dispositif"]["id"]);
        if (!empty($obj_dispositif->id)) {
            $obj_dispositif->load_relationship("ops_dispositif_ops_dossier");
            $obj_dispositif->ops_dispositif_ops_dossier->add($dossier_id);
        }

        return (!empty($dossier_id)) ? $dossier_id : "";
    }

    /**
     * @access public
     * @name updateDossier()
     * Fonction pour met à jour un dossier
     *
     *  @param string                         $dossier_id : L'id du dossier à modifier
     *  @param array                           $formulaire : Tableau des champs du formulaire
     *  @return string|boolean               $dossier_id : retourne l'id du dossier crée sinon false
     */
    public static function updateDossier($formulaire, $dossier_id)
    {
        $obj_dossier = BeanFactory::getBean('OPS_dossier', $dossier_id);
        if (empty($obj_dossier->id)) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Le dossier id = " . $dossier_id . " n'a pas pu etre récupéré. ");
            return false;
        }

        if (empty($obj_dossier->type_tiers)) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Le dossier id = " . $obj_dossier->id . " ne dispose pas d'un type tiers. ");
            return false;
        } else {
            $formulaire["type_tiers"] = $obj_dossier->type_tiers;
        }

        // Vérification des champs indispensable pour la création d'un dossier sur OpenSub
        if (OPS_dossier::isFormulaireValid($formulaire) === false) {
            return false;
        }

        // On récupere les champs de la vue associé au dispositif
        $DispositifModel = new DispositifModel($formulaire['dispositif']);
        $champs_vue = $DispositifModel->getChampsVueAgent();

        // Si les champs sont vide, on retourne false
        if (!is_array($champs_vue) || count($champs_vue) === 0) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Les champs de la vue agent associé au dispositif n'ont pas pu etre récupéré. ");
            return false;
        }

        // On initialise les champs à créer ( Divise les champs en deux tableau basic / custom )
        $dossier_champs = OPS_dossier::getDataDossier($formulaire, $champs_vue);
        if (!is_array($dossier_champs) || count($dossier_champs) === 0) {
            $GLOBALS['log']->fatal(" OPS_dossier :: updateDossier => Echec de l'initialisation des champs dossier => dossier non crée . ");
            return false;
        }

        unset($dossier_champs["basic"]["type_tiers"]);

        // On crée le dossier sur la base de champs effectif préremplis (fournis via formulaire vs champs disponible|possibles de la vue assiciée)
        $dossier_id = OPS_dossier::editBeanDossier($obj_dossier, $dossier_champs);

        return (!empty($dossier_id)) ? $dossier_id : false;
    }

    /**
     * @access private
     * @name isFormulaireValid()
     *
     * Fonction qui retourne apres vérification la valeur d'un champ de type "BoutonRadio"
     *
     *  @param array                    $formulaire: Les champs retournés par le formulaire
     *  @return boolean                 $statut_vérification : true ou false
     */
    private static function isFormulaireValid($formulaire)
    {

        // Si le type de la variable $champs_formulaire est différent de array
        if (is_array($formulaire) === false) {
            $GLOBALS['log']->fatal(" OPS_dossier :: isFormulaireValid => Les données recu pour la création du dossier ne sont pas au format Array ");
            return false;
        }

        // Si le tableau $champs_formulaire est vide
        if (count($formulaire) == 0) {
            $GLOBALS['log']->fatal(" OPS_dossier :: isFormulaireValid => Le tableau recu pour la création du dossier est vide  ");
            return false;
        }

        // Si on ne dispose pas du champ type_tiers
        if (empty($formulaire['type_tiers'])) {
            $GLOBALS['log']->fatal(" OPS_dossier :: isFormulaireValid => Le champ 'type tiers' est absent ou vide du tableau de création d'un dossier ");
            return false;
        }

        // Si on ne dispose pas du champ dispositif ou vide
        if (empty($formulaire['dispositif'])) {
            $GLOBALS['log']->fatal(" OPS_dossier :: isFormulaireValid => Le champ 'dispositif' est absent ou vide  du tableau de création d'un dossier ");
            return false;
        }

        // Si on ne dispose pas du champ demandeur ou vide
        if (empty($formulaire['demandeur'])) {
            $GLOBALS['log']->fatal(" OPS_dossier :: isFormulaireValid => Le champ 'demandeur' est absent ou vide du tableau de création d'un dossier ");
            return false;
        }

        return true;
    }

    /**
     * @access private
     * @name initDataDossier()
     * Fonction qui génére les boutons d'onglets
     *
     *  @param array            $onglets : la liste des onglets rattaché à la vue
     *  @return string        $btn_onglets_html : code html , les boutons sont groupés dans une div
     */
    //                                 $champs_formulaire, $champs_vue
    private static function initDataDossier($champs_formulaire, $champs_onglets)
    {


        $erreur = false;
        $champs_dossier = array(
            'basic' => array(),
            'custom' => array(),
            'unknow' => array()
        );

        $demandeur_type = $champs_formulaire["type_tiers"];

        // On parcours les champs de la vue du dispositif associée à ce dossier...
        foreach ($champs_onglets as $field_slug_id => $field_params) {

            // Définition du type de champ [basic | custom]
            $field_type = $field_params['description'];

            // On récupère la valeur par défaut du champ..
            if(!empty($field_params) && !empty($field_params['type']))
            {
                $field_default_value = OPS_dossier::getChampDefautValue($field_params);
            }
            else
            {
                $GLOBALS['log']->fatal('OPS_dossier::initDataDossier => Le type du champ "' . $field_slug_id . '" est vide' );
                continue;
            }

            // Si le champ existe dans les champs du formulaire (envoyé), sinon affecte la valeur par défaut.
            if (array_key_exists($field_slug_id, $champs_formulaire)) {

                // La valeur transmise doit être conforme :
                // - Fournie si obligatoire, sinon on utilise la valeur par défaut
                if (empty($champs_formulaire[$field_slug_id]) && $field_params['obligatoire'] === 1) {

                    $champs_dossier[$field_type][$field_slug_id] = $field_default_value;
                }

                // #911 Gestion du cas ou la valeur du champ de type "checkbox" est vide [ Création par le connecteur ]
                if (empty($champs_formulaire[$field_slug_id]) && $field_params['type'] === "checkbox") {
                    $champs_formulaire[$field_slug_id] = "0";
                }

                // On récupère la valeur issue du type de champ
                if (!empty($champs_formulaire[$field_slug_id]) || strval($champs_formulaire[$field_slug_id]) === "0") {

                    $field_correct_value = OPS_dossier::getChampValueByType($champs_formulaire[$field_slug_id], $field_params, $demandeur_type);

                    if ($field_correct_value !== false) { //

                        $champs_dossier[$field_type][$field_slug_id] = $field_correct_value; // On effecte la nouvelle valeur récoltée.

                    } else {
                        $GLOBALS['log']->fatal(" OPS_dossier :: isFormulaireValid =>  Echec lors de la récupération de la valeur du champ [" . $field_params['libelle'] . "] de type « " . $field_params['type'] . " »");
                        $erreur = true;
                    }
                } else {
                    $champs_dossier[$field_type][$field_slug_id] = $field_default_value;
                }
            }
            // Ce champ est absent (non envoyé), donc affetcte la valeur par défaut.
            else {

                $champs_dossier[$field_type][$field_slug_id] = $field_default_value;
            }
        }


        // A présent il faut néanmoins préserver les champs transmis absent dans la vue :
        // - Mais presents dans le référentiel -> a classer dans [basic | custom]
        // - Mais absents du référentiel -> a classer dans [unknow]

        $referentiel_fields = OPS_generateur_referentiel::getListReferentiel();
        $referentiel_fields_slugs = array_column($referentiel_fields, 'name');


        // On parcours les champs transmis via le formulaire, certains ne sont pas disponibles dans la vue associée...
        foreach ($champs_formulaire as $field_slug_id => $field_value) {

            // Ce champ existe t-il dans le référentiel ?
            $field_index = array_search($field_slug_id, $referentiel_fields_slugs);
            if ($field_index !== false) {

                // On se simplifie avec ses params...
                $field_params = $referentiel_fields[$field_index];

                // Définition du type de champ [basic | custom]
                $field_type = $field_params['description'];

                // On récupère la valeur par défaut du champ..
                $field_default_value = OPS_dossier::getChampDefautValue($field_params);

                if (isset($champs_dossier[$field_type][$field_slug_id])) {
                    // Ce champ existe déjà dans le flow de champs, il a donc été mappé via la vue :)
                } else {

                    // A ce stade, le champ transmis via le formulaire n'est pas présent dans la composition du formulaire de vue, mais existe dans le référentiel...
                    // On va donc le récupérer correctement.

                    // La valeur transmise doit être conforme :
                    // - Fournie si obligatoire, sinon on utilise la valeur par défaut
                    if (empty($champs_formulaire[$field_slug_id]) && $field_params['obligatoire'] === 1) {

                        $champs_dossier[$field_type][$field_slug_id] = $field_default_value;
                    }

                    // On récupère la valeur issue du type de champ
                    if (!empty($champs_formulaire[$field_slug_id]) || strval($champs_formulaire[$field_slug_id]) === "0") {

                        $field_correct_value = OPS_dossier::getChampValueByType($champs_formulaire[$field_slug_id], $field_params, $demandeur_type);

                        if ($field_correct_value !== false) { //

                            $champs_dossier[$field_type][$field_slug_id] = $field_correct_value; // On effecte la nouvelle valeur récoltée.

                        } else {
                            $GLOBALS['log']->fatal(" OPS_dossier :: isFormulaireValid =>  Echec lors de la récupération de la valeur du champ [" . $field_params['libelle'] . "] de type « " . $field_params['type'] . " »");
                            $erreur = true;
                        }
                    } else {
                        $champs_dossier[$field_type][$field_slug_id] = $field_default_value;
                    }
                }
            }
        }


        return ($erreur !== true) ? $champs_dossier : [];
    }

    /**
     * @access private
     * @name getChampDefautValue()
     * Fonction qui retourne la valeur par défaut du champ sinon empty
     *
     *  @param array            $champ : Parametres du champ dans le générateur
     *  @return string          $valeur_defaut : la valeur par défaurt du champ sinon empty
     */
    private static function getChampDefautValue($champ)
    {
        $defaut_value = "";
        if (!empty($champ['defaut'])) {
            $fieldFactory = new OdeFieldFactory();
            $defaut_value = $fieldFactory->getChampDefautValue($champ);
        }
        return $defaut_value;
    }

    /**
     * @access private
     * @name getChampValueByType()
     * Fonction qui retourne la valeur du champ selon le type
     *
     *  @param array            $champ_formulaire_valeur : la valeur du champ dans le formulaire
     *  @param array            $champ_onglet : Définiation du champ dans la vue dossier
     *  @param array            $conditions : liste des conditions
     *  @return string|boolean|array            $value : valeur du champ
     */
    private static function getChampValueByType($champ_formulaire_valeur, $champ_onglet, $demandeur_type)
    {
        if ($champ_onglet['type'] === "relation") {
            return OPS_dossier::getChampRelationValue($champ_formulaire_valeur, $champ_onglet, $demandeur_type);
        }

        if (in_array($champ_onglet['type'], ["text", "text_long", "wysiwyg", "bic", "iban", "adresse", "api_particulier", "adresse_geoloc", "nir","api_r2p"])) {
            return (is_string($champ_formulaire_valeur) && !empty($champ_formulaire_valeur)) ? $champ_formulaire_valeur : false;
        }

        if (in_array($champ_onglet['type'], ["montant", "nombre"])) {
            return str_replace(',', '.', str_replace(' ', '', $champ_formulaire_valeur));
        }

        if ($champ_onglet['type'] === "date") {
            return (!empty($champ_formulaire_valeur)) ? OdeDateHelper::toSQL($champ_formulaire_valeur) : "";
        }

        if (in_array($champ_onglet['type'], ["lien", "pourcentage", "checkbox", "liste", "radio", "number"])) {
            return $champ_formulaire_valeur;
        }

        if ($champ_onglet['type'] === "file") {
            return (!empty($champ_formulaire_valeur)) ? $champ_formulaire_valeur : "";
        }

        return $champ_formulaire_valeur;
    }



    /**
     * @access private
     * @name getChampRelationValue()
     * Fonction qui retourne apres vérification la valeur d'un champ de type "relation"
     *
     *  @param array                          $champ_formulaire_valeur: La valeur du champ retourné par le formulaire
     *  @param array                          $champ_onglet: La définition du champ dans la vue dossier
     *  @return string|array|boolean          $champ_valeur: La valeur du champ vérifié ou false
     */
    private static function getChampRelationValue($champ_formulaire_valeur, $champ_onglet, $demandeur_type)
    {
        $module = [];
        $params_decoded = OdeArrayHelper::jsonToArray(base64_decode($champ_onglet['params']));
        if ($params_decoded['statut'] == "ok") {
            $params = $params_decoded['data'];
            if (!empty($champ_onglet["name"]) && $champ_onglet["name"] === "demandeur") {
                if (!empty($demandeur_type)) {
                    foreach ($params["modules"] as $_module) {
                        if ($_module["name"] === $demandeur_type) {
                            $module = [
                                "module_name" => $_module["name"],
                                "module_id" => $champ_formulaire_valeur,
                                "relation_name" => $_module["relation"],
                                "unique" => $params["unique"],
                                "lien" => $params["lien"],
                                "type" => $params["type"]
                            ];
                        }
                    }
                }
            } else {
                $module = $params;
                $module["module_id"] = $champ_formulaire_valeur;
            }
        } else {
            $GLOBALS['log']->fatal(" OdeRelation :: getModuleField() =>  Erreur decodage params " . $params_decoded['data']);
        }
        return (is_array($module) && count($module) > 0) ? $module : false;
    }

    /**
     * @access private
     * @name getChampVidebyType()
     * Fonction qui retourne la valeur du champ selon le type
     *
     *  @param array            $champ_onglet : Définiation du champ dans la vue dossier
     *  @return string|boolean|array            $value : valeur du champ
     */
    private static function getChampVidebyType($champ_onglet, $demandeur_type)
    {
        $value = "";

        if (in_array($champ_onglet['type'], ["numerique", "checkbox"])) {
            return 0;
        }

        if ($champ_onglet['type'] === "relation") {
            return OPS_dossier::getChampRelationValue("", $champ_onglet, $demandeur_type);
        }

        if (in_array($champ_onglet['type'], ["pourcentage", "montant", "number"])) {
            return "";
        }

        return $value;
    }

    /*******************************************************************************************************************************************/
    /*******************************************************************************************************************************************/
    /*******************************************************************************************************************************************/








    /**
     * @access private
     * @name setFirstStatut()
     * Fonction qui défini le premier statut (initialisation), crée une historisation et déclenche les actions statut
     *
     *
     *  @return boolean    true / false
     */
    private static function setFirstStatut($obj_dossier, $nouveau_statut = null)
    {

        global $current_user;

        $statut_updated = false;

        // On récupere l'objet OPS_statut
        $obj_statut = ($nouveau_statut !== null) ? BeanFactory::getBean('OPS_statut', $nouveau_statut) : false;

        // On récupere l'id de la nouvelle étape associé au statut de démarrage
        $liste_etapes = (!empty($obj_statut->id)) ? $obj_statut->get_linked_beans('ops_etape_ops_statut', 'OPS_etape') : false;
        $obj_etape = (is_array($liste_etapes) && count($liste_etapes) == 1) ? $liste_etapes[0] : false;

        if (!empty($obj_etape->id)) {

            if (!empty($obj_statut->id) && !empty($obj_etape->id)) {

                // On déclenche les actions statut associées
                $obj_statut->action_statut($obj_dossier);

                $ojbDossier = BeanFactory::getBean('OPS_dossier', $obj_dossier->id);

                // On met à jour le statut et l'étape du dossier
                $obj_dossier->incomplet = $obj_statut->dossier_incomplet;
                $obj_dossier->ops_etape_id = (!empty($ojbDossier->ops_etape_id)) ? $ojbDossier->ops_etape_id : $obj_etape->id;
                $obj_dossier->ops_statut_id = (!empty($ojbDossier->ops_statut_id)) ? $ojbDossier->ops_statut_id : $obj_statut->id;

                // On met à jour la date d'avancement du dossier
                $timedate = new TimeDate($current_user);
                $date = $timedate->getNow(true);
                $date = $date->format("Y-m-d");
                $obj_dossier->date_avancement = $date;
                $obj_dossier->date_transmission = $date;

                // Historisation du nouveau statut
                $histo = new OPS_historisation();
                $histo->canal = "action_unitaire";
                $histo->ops_etape_id = $obj_etape->id;
                $histo->ops_statut_id = $obj_statut->id;
                $histo->auteur = $current_user->name;
                $histo->flag_retour = '1';
                $id_histo = $histo->save(false);

                if (!empty($id_histo)) {
                    $obj_dossier->load_relationship('ops_historisation_ops_dossier');
                    $obj_dossier->ops_historisation_ops_dossier->add($id_histo);
                    return $obj_dossier;
                } else {
                    $GLOBALS['log']->fatal(" OPS_dossier :: setFirstStatut => Echec de création historisation sur dossier id = " . print_r($obj_dossier->id, true));
                }
            } else {
                $GLOBALS['log']->fatal(" OPS_dossier :: setFirstStatut => Echec de l'initialisation de l'étape et du statut du dossier id = " . print_r($obj_dossier->id, true));
            }
        } else {
            $GLOBALS['log']->fatal(" OPS_dossier :: setFirstStatut = >Dossier introuvable, Initialiser le dossier avant d'appeler la fonction :: dossier->setFirstStatut( Id du nouveau statut ) ");
        }

        return $obj_dossier;
    }


    /**
     * @access public
     * @name updateStatut()
     * Fonction qui met à jour le statut , crée une historisation et déclenche les actions statut
     *
     *
     *  @return boolean    true / false
     */

    public function updateStatut($nouveau_statut = null, $type_modification = null, $is_action = null,$auteur = '')
    {

        global $current_user, $beanFiles;

        $module = $this->module_dir;
        $bean = $this;
        require_once($beanFiles["OPS_etape"]);
        require_once($beanFiles["OPS_statut"]);
        require_once($beanFiles["OPS_historisation"]);

        $statut_updated = false;

        if (!empty($nouveau_statut)) {

            $timedate = new TimeDate($current_user);
            $date = $timedate->getNow(true);
            $date = $date->format("Y-m-d");

            if ($nouveau_statut !== "precedent") {

                // Initialisation du nouveau statut
                $this->ops_statut_id = $nouveau_statut;
                $statut_updated = true;



            } else {

                // Récupération des status passés
                $tab_historiques = $this->get_linked_beans('ops_historisation_ops_dossier', 'OPS_historisation', '', 0, -1, 0, "flag_retour = '1'");
                $last_histo = end($tab_historiques);

                foreach ($tab_historiques as $key => $historisation) {

                    $obj_historisation = new OPS_historisation();
                    $obj_historisation->retrieve($historisation->id);


                    $uneEtape = new OPS_etape();
                    $uneEtape->retrieve($obj_historisation->ops_etape_id);

                    $etapeCourante = new OPS_etape();
                    $etapeCourante->retrieve($this->ops_etape_id);

                    // On exclu du tableau les statuts des étapes suivantes ainsi que le statut actuel
                    if ($this->ops_statut_id != $obj_historisation->ops_statut_id && $uneEtape->ordre <= $etapeCourante->ordre) {

                        $date = $obj_historisation->date_entered;
                        $formDate = substr($date, 6, 4) . "-" . substr($date, 3, 2) . "-" . substr($date, 0, 2) . " " . substr($date, 11, 7);
                        $tab_statut_prec[$formDate]['id_statut'] = $obj_historisation->ops_statut_id;
                        $tab_statut_prec[$formDate]['id_histo'] = $obj_historisation->id;
                    }

                    if ($this->ops_statut_id == $obj_historisation->ops_statut_id) {

                        $obj_historisation->flag_retour = "0";
                        $obj_historisation->save();
                    }
                }

                ksort($tab_statut_prec);

                foreach ($tab_statut_prec as $key => $value) {

                    $ancienHisto = $value;
                }

                // Flag sur l'ancien histo du statut précédent
                $obj_historisation = new OPS_historisation();
                $obj_historisation->retrieve($ancienHisto['id_histo']);
                $obj_historisation->flag_retour = "0";
                $obj_historisation->save();

                // Initialisation du nouveau statut
                $this->ops_statut_id = $ancienHisto['id_statut'];
                
            }

            $statut = new OPS_statut();
            $unStatut = $statut->retrieve($this->ops_statut_id);

            $etapes = $unStatut->get_linked_beans('ops_etape_ops_statut', 'OPS_etape');
            foreach ($etapes as $etape) { // Actuellement 1 seule etape possible
            }

            $etapeCourante = new OPS_etape();
            $etapeCourante->retrieve($this->ops_etape_id);

            $cancel_action = false;
            // On exclu du tableau les statuts des étapes suivantes ainsi que le statut actuel
            if ($etapes[0]->ordre < $etapeCourante->ordre) {
                $cancel_action = true;
            }


            $this->ops_etape_id = $etape->id;
            $this->date_avancement = $date;

            $obj_statut_flag = new OPS_statut();
            $statut_courant = $obj_statut_flag->retrieve($this->ops_statut_id);

            $this->cloture = $statut_courant->cloture;

            $this->incomplet = $statut_courant->dossier_incomplet;
            $testy = $this->save();

            // Historisation du nouveau statut
            $uneHistorisation = new OPS_historisation();
            $uneHistorisation->canal = ($type_modification === "mass") ? "action_masse" : "action_unitaire";
            $uneHistorisation->ops_etape_id = $etape->id;
            $uneHistorisation->ops_statut_id = $this->ops_statut_id;
            $uneHistorisation->auteur = (!empty($auteur))?$auteur:$current_user->name;;
            $uneHistorisation->flag_retour = '1';

            // Création de la relation avec le module OPS_Subvention
            $uneHistorisation->ops_historisation_ops_dossier_name = $this->name;
            $uneHistorisation->ops_dossier_id = $this->id;

            // Flagé l'historisation si elle existe deja pour la meme etape et meme statut
            $tabs_historiques = $this->get_linked_beans('ops_historisation_ops_dossier', 'OPS_historisation', '', 0, -1, 0, "flag_retour = '1'");
            foreach ($tabs_historiques as $key => $value) {

                $ancienHisto = $value;
            }
            #   $ancienHisto = array_pop( $tabs_historiques ); //

            $old_historisation = new OPS_historisation();
            $old_historisation->retrieve($ancienHisto->id);

            if ($uneHistorisation->ops_etape_id === $old_historisation->ops_etape_id && $uneHistorisation->ops_statut_id === $old_historisation->ops_statut_id) {
                $old_historisation->flag_retour = "0";
                $old_historisation->save();
            }

            // Sauvegarde de l'historisation
            $uneHistorisation->save();

            $statut_updated = true;
        }

        // Déclenchement de l'action au changement de statut, action_statut dans OPS_statut
        if (empty($is_action) && $statut_updated && !$cancel_action) {
            $obj_statut = new OPS_statut();
            $obj_statut->retrieve($this->ops_statut_id);
            $test = $obj_statut->action_statut($this);
        }

        return $statut_updated;
    }

    /**
     * @access private
     * @name editBeanDossier()
     * Fonction qui crée l'objet dossier
     *
     *  @param array            $onglets : la liste des onglets rattaché à la vue
     *  @return string        $btn_onglets_html : code html , les boutons sont groupés dans une div
     */

    private static function editBeanDossier($obj_dossier, $champs_dossier, $save_brl = false)
    {
        global $db;

        $champs_exclus = ["modified_user_id", "created_by", "assigned_user_id"];

        $relations = [];
        foreach ($champs_dossier['basic'] as $champ_name => $champ_value) {

            $champ_value = (!is_array($champ_value)) ? stripslashes($champ_value) : $champ_value;

            // On exclus les champs de base qui ont des valeurs par defaut et les valeurs vides
            if (!in_array($champ_name, $champs_exclus)) {

                if (!is_array($champ_value)) {
                    //On traitre le champ liste simple
                    if ($obj_dossier->field_name_map[$champ_name]['type'] == 'enum' && substr_count($champ_value, '^') == 2) {
                        $champ_value = str_replace("^", "", $champ_value);
                    }
                    // Le champ n'est pas de type relation => insertion direct
                    $obj_dossier->$champ_name = $champ_value;
                } else if (!empty($champ_value['type']) && $champ_value['type'] == "parent_id") {
                    // Le champ est de type relation mais avec un parent_id => insertion direct
                    $obj_dossier->$champ_name = $champ_value['module_id'];
                } else {

                    $insertion_direct = array("OPS_personne_morale", "OPS_individu", "OPS_dispositif", "OPS_campagne", "OPS_exercice", "OPS_elu");
                    if (in_array($champ_value['module_name'], $insertion_direct)) {
                        $obj_dossier->{strtolower($champ_value['module_name']) . "_id"} = $champ_value['module_id'];
                    }

                    // OPS_elu
                    if ($champ_value['module_name'] === "OPS_elu") {
                        $elus = (!empty($obj_dossier->id)) ? $obj_dossier->get_linked_beans("ops_elu_ops_dossier", "OPS_elu") : array();
                        if (is_array($elus) && count($elus) > 0) {
                            foreach ($elus as $elu) {
                                $obj_dossier->load_relationship("ops_elu_ops_dossier");
                                $obj_dossier->ops_elu_ops_dossier->delete($obj_dossier->id, $elu);
                            }
                        }
                        $module_ids = array();
                        if ($champ_value['unique'] === "multi" && strpos($champ_value['module_id'], '|') !== false) {
                            $champ_module_ids = explode("|", $champ_value['module_id']);
                            foreach ($champ_module_ids as $champ_module_id) {
                                if (!empty($champ_module_id)) {
                                    $module_ids[] = $champ_module_id;
                                }
                            }
                        } else {
                            $module_ids[] = $champ_value['module_id'];
                        }
                        foreach ($module_ids as $module_id) {
                            $obj_dossier->load_relationship("ops_elu_ops_dossier");
                            $obj_dossier->ops_elu_ops_dossier->add($module_id);
                        }
                    }

                    // OPS_reunion
                    if ($champ_value['module_name'] === "OPS_reunion") {
                        $relations[] = $champ_value;
                    }

                    // OPS_commision
                    if ($champ_value['module_name'] === "OPS_commission") {
                        $commissions = (!empty($obj_dossier->id)) ? $obj_dossier->get_linked_beans("ops_commission_ops_dossier", "OPS_commission") : array();
                        if (is_array($commissions) && count($commissions) > 0) {
                            foreach ($commissions as $commission) {
                                $obj_dossier->load_relationship("ops_commission_ops_dossier");
                                $obj_dossier->ops_commission_ops_dossier->delete($obj_dossier->id, $commission);
                            }
                        }
                        $obj_dossier->load_relationship("ops_commission_ops_dossier");
                        $obj_dossier->ops_commission_ops_dossier->add($champ_value['module_id']);
                    }

                    // OPS_sous_territoire
                    if ($champ_value['module_name'] === "OPS_sous_territoire") {
                        $sousTerritoires = (!empty($obj_dossier->id)) ? $obj_dossier->get_linked_beans("ops_sous_territoire_ops_dossier", "OPS_sous_territoire") : array();
                        if (is_array($sousTerritoires) && count($sousTerritoires) > 0) {
                            foreach ($sousTerritoires as $sousTerritoire) {
                                $obj_dossier->load_relationship("ops_sous_territoire_ops_dossier");
                                $obj_dossier->ops_sous_territoire_ops_dossier->delete($obj_dossier->id, $sousTerritoire);
                            }
                        }
                        $obj_dossier->load_relationship("ops_sous_territoire_ops_dossier");
                        $obj_dossier->ops_sous_territoire_ops_dossier->add($champ_value['module_id']);
                    }

                    // OPS_dossier
                    if ($champ_value['module_name'] === "OPS_dossier") {

                        $dossiers = array();
                        $sql = " SELECT `ops_dossier_id` FROM `ops_dossier_ops_dossier` WHERE `ops_dossier_id2` = '" . $relationId . "' AND `deleted` = '0' ";
                        $result = $db->query($sql);
                        while ($row = $db->fetchByAssoc($result)) {
                            $dossiers[] = $row['ops_dossier_id'];
                        }
                        if (is_array($dossiers) && count($dossiers) > 0) {
                            foreach ($dossiers as $dossier) {
                                $dossierBean = BeanFactory::getBean('OPS_dossier', $dossier);
                                $obj_dossier->load_relationship("ops_dossier_ops_dossier");
                                $obj_dossier->ops_dossier_ops_dossier->delete($obj_dossier->id, $dossierBean);
                            }
                        }
                        $relationId = str_replace("^", "", $champ_value['module_id']);
                        $relationId = trim($relationId);
                        $dossier_bean = BeanFactory::getBean('OPS_dossier',$relationId);
                        $dossier_bean->load_relationship("ops_dossier_ops_dossier");
                        $dossier_bean->ops_dossier_ops_dossier->add($obj_dossier->id);
                        
                    }

                    // OPS_contrat
                    if ($champ_value['module_name'] === "OPS_contrat") {
                        $contrats = (!empty($obj_dossier->id)) ? $obj_dossier->get_linked_beans("ops_contrat_ops_dossier", "OPS_contrat") : array();
                        if (is_array($contrats) && count($contrats) > 0) {
                            foreach ($contrats as $contrat) {
                                $obj_dossier->load_relationship("ops_contrat_ops_dossier");
                                $obj_dossier->ops_contrat_ops_dossier->delete($obj_dossier->id, $contrat);
                            }
                        }
                        $relationId = str_replace("^", "", $champ_value['module_id']);
                        $relationId = trim($relationId);
                        $obj_dossier->load_relationship("ops_contrat_ops_dossier");
                        $obj_dossier->ops_contrat_ops_dossier->add($relationId);
                        $obj_dossier->ops_contrat_id = $relationId;
                    }

                }

            }
        }

        //On ajoute les justificatfs, on envoie un lot
        $champs_dossier = OPS_dossier::addJustificatif($obj_dossier, $champs_dossier);

        if(isset($champs_dossier['flag']['justificatif']) && !empty($champs_dossier['flag']['justificatif']))
        {
            $obj_dossier->flag_justificatif_etude = $champs_dossier['flag']['justificatif'];
        }

        //On exécute les champs fonction
        $champs_dossier = OPS_dossier::executeFonctions($obj_dossier, $champs_dossier);
        $champs_dossier = OPS_dossier::executeGrille($obj_dossier, $champs_dossier);

        // On récpère les champs déjà existant avec leur valeurs, qu'on va merger avec les nouvelles valeurs transmises.
        $old_champs_custom = (isset($obj_dossier->champs_custom) && !empty($obj_dossier->champs_custom)) ? json_decode(base64_decode($obj_dossier->champs_custom), true) : array();
        $new_champs_custom = isset($champs_dossier['custom']) ? $champs_dossier['custom'] : array();

        if(is_array($old_champs_custom))
        {
            // Tous les champs custom du formulaire de la vue doivent être soit fournis soit remplis avec les valeurs existantes.
            $champs_custom = array_merge($old_champs_custom, $new_champs_custom);
        }
        else
        {
            $champs_custom = $new_champs_custom;
        }

        foreach ($champs_custom as $key => $value) {
            if (is_string($value)) {
                $champs_custom[$key] = stripslashes($value);
            }
        }

        $obj_dossier->champs_custom = base64_encode(json_encode($champs_custom));

        // Si on a cliqué sur "Enregistrer et revenir plus tard" ou que la demande est sortie du statut brouillon, on enlève le nom "TEMP" du dossier
        if ($save_brl || $obj_dossier->brouillon == "non") {
            $obj_dossier->name = (!empty($obj_dossier->libelle_dossier)) ? $obj_dossier->num_dossier . " - " . $obj_dossier->libelle_dossier : $obj_dossier->num_dossier;
        }

        if ($obj_dossier->save()) {
            if (is_array($relations) && count($relations) > 0) {
                foreach ($relations as $relation) {
                    ReunionModel::deleteRelations($obj_dossier->id);
                    if (!empty($relation['module_id'])) {
                        ReunionModel::addRelationDossier($obj_dossier->id, $relation['module_id']);
                    }
                }
            }
            return $obj_dossier->id;
        } else {
            return false;
        }
    }

    /**
     * @access private
     * @name addJustificatif()
     * Fonction qui enregistre les justificatifs
     *
     *  @param array            $obj_dossier : dossier
     * @param array            $champs_dossier : Champs du dossier
     *  @return array        $champs_dossier : Champs du dossier
     */
    private static function addJustificatif($obj_dossier, $champs_dossier){

        global $sugar_config;

        require_once __DIR__ . '/ScanPJ.php';
   //     require_once($_SERVER['DOCUMENT_ROOT'] . '/vendor/setasign/fpdi/src/autoload.php');

        if(empty($sugar_config['ops_justificatif']['chemin'])){
            $GLOBALS['log']->fatal(print_r("addJustificatif : le chemin de stockage n'a pas été paramétré",true));
        }

        $obj_dossier->load_relationship("ops_dossier_ops_justificatif");

        if (!empty($obj_dossier->ops_personne_morale)) {
            $obj_personne_morale = BeanFactory::getBean('OPS_personne_morale', $obj_dossier->ops_personne_morale);

            if(!empty($obj_personne_morale->id))
            {
                $obj_personne_morale->load_relationship("ops_personne_morale_ops_justificatif");
            }
        }

        if(!empty($obj_dossier->champs_custom)) {
            $customChamp = json_decode( base64_decode( $obj_dossier->champs_custom) );
        }


        foreach ($champs_dossier['custom'] as $key => $value) {

            # Evolution #1372 [Traitement File provenant du front sous forme linéarisée]
            // Le champ peut être en provenance du formulaire front (étape par étape) et encodé en b64/json.
            // Cas de la réutilisation d'un document, où le content vaudrait :
            // - Soit le chemin de la PJ déjà déposée (donc déjà enregistrée).
            // - Soit le contenu brut b64 du fichier !
            // -------------------------------------------------------------------------
            /**
             * On peut détecter la longueur minimale autorisée du base 64 dans notre système:
             * 
             * # Cas d'un fichier image de 1px réel brut transmis ( à minima ):
             * 
             * Base64 du binaire d'un tel fichier: iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+P+/HgAFhAJ/wlseKgAAAABJRU5ErkJggg==
             * 96 octets
             * 
             * En le codant sous forme JSON (phase d'encodage dans le formulaire fron PRE envoi), avec un nom minimal, une taille sur 1 caractère, son ID sur 36 octets :
             * 
             * {
             *  "name":"1.png",
             *  "type":"image/png",
             *  "size":1,
             *  "attache":"aaaaaaaa-aaaa-aaaa-aaaa-aaaaaaaaaaaa",
             *  "content":"iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+P+/HgAFhAJ/wlseKgAAAABJRU5ErkJggg=="
             * }
             * 
             * On obtient une fois encodé en B64 :
             * eyJuYW1lIjoiMS5wbmciLCJ0eXBlIjoiaW1hZ2UvcG5nIiwic2l6ZSI6MSwiYXR0YWNoZSI6ImFhYWFhYWFhLWFhYWEtYWFhYS1hYWFhLWFhYWFhYWFhYWFhYSIsImNvbnRlbnQiOiJpVkJPUncwS0dnb0FBQUFOU1VoRVVnQUFBQUVBQUFBQkNBWUFBQUFmRmNTSkFBQUFEVWxFUVZSNDJtTmsrUCsvSGdBRmhBSi93bHNlS2dBQUFBQkpSVTVFcmtKZ2dnPT0ifQ==
             * Soit 272 chars/octets
             * 
             * -------------------------------------------------
             * 
             * # Cas d'un justificatif avec le chemin d'une ressource déjà enregistrée :
             * 
             * Son contenu n'est pas encodé en base64 car il correspond au chemin de la pièce justificatve déjà existante.
             * 
             * On le code de la même façon qu'un fichier brut transmis à l'exception qu'on injecte le chemin dans le content plutôt que binaryB64Coded du fichier...
             * Les UUID du dossier/personne morale, et celui de la PJ sont codées sous 36 caractères...
             * 
             * {
             *  "name":"1.jpg",
             *  "type":"image/jpeg",
             *  "size":"4"
             *  "attache":"aaaaaaaa-aaaa-aaaa-aaaa-aaaaaaaaaaaa",
             *  "content":"upload/justificatifs/aaaaaaaa-aaaa-aaaa-aaaa-aaaaaaaaaaaa/aaaaaaaa-aaaa-aaaa-aaaa-aaaaaaaaaaaa"
             * }
             * 
             * On obtient une fois encodé en B64 :
             * eyJuYW1lIjoiMS5qcGciLCJ0eXBlIjoiaW1hZ2UvanBlZyIsInNpemUiOiIxIiJhdHRhY2hlIjoiYWFhYWFhYWEtYWFhYS1hYWFhLWFhYWEtYWFhYWFhYWFhYWFhIiwiY29udGVudCI6InVwbG9hZC9qdXN0aWZpY2F0aWZzL2FhYWFhYWFhLWFhYWEtYWFhYS1hYWFhLWFhYWFhYWFhYWFhYS9hYWFhYWFhYS1hYWFhLWFhYWEtYWFhYS1hYWFhYWFhYWFhYWEifQ==
             * Soit 272 chars/octets
             * 
             * A minima la longueur du content diffère, mais codé en Base64, le modulo final permet d'obtenir dans les 2 cas (hasard) une longueur de 272 caractères.
             **/
            if (!defined('MIN_JSON_B64_LENGTH')) define('MIN_JSON_B64_LENGTH', 272);

            # Identifie si la valeur est potentiellement un base64 de type fichier.
            if ( !is_array($value) && strlen($value) >= MIN_JSON_B64_LENGTH ) {
                // Probably a string and B64 Coded => A streamlined file from a JSON !
                $objFile = json_decode(base64_decode($value), false, 512, JSON_INVALID_UTF8_IGNORE);
                if( isset($objFile->name) 
                 && isset($objFile->type)
                 && isset($objFile->size)
                 && isset($objFile->attache)
                 && isset($objFile->content) && trim($objFile->content) != '' )
                {
                    $value = (array) $objFile; 
                }
            }


            # Assemblage en cas de fichiers multiple 
            if( isset($value['attache']) && !empty($value['attache'])
             && isset($value['multiple']) && $value['multiple'] == "multiple"
             && is_array($value['files']) )
            {
                $piece_justificative_pdf_files = [];
                foreach ($value['files'] as $file) {

                    $file = (array) $file;
                 
                    // Before conversion, we store physically the file.
                    $tmp_name = $uuid .'_'. md5( $file['name'] ) . date('YmdHis');

                    $chemin_source = $sugar_config['upload_dir'].$tmp_name ; 
            
                    if( file_put_contents( $chemin_source, base64_decode( $file['content'] ) ) ){

                        // Sécurité - Scan 
                        $obj_scan = new ScanPJ();
                        $chemin_verif = $_SERVER['DOCUMENT_ROOT'] .'/'. $chemin_source ; 
                        $ret = $obj_scan->scanPathForMalware( $chemin_verif );
                        

                        if( strtolower($file['type']) === 'application/pdf' ){
                            rename( $chemin_source , $chemin_source.'.pdf' );
                            $piece_justificative_pdf_files[] = $chemin_source.'.pdf';
                        }else{

                            switch ( strtolower($file['type']) ) {
                                case 'image/jpeg':
                                    shell_exec('convert '.$chemin_source.' '.$chemin_source.'.png');
                                    shell_exec('convert '.$chemin_source.'.png '.$chemin_source.'.pdf'); 
                                    if (file_exists($chemin_source.'.png')) { unlink( $chemin_source.'.png' ); }      
                                    break;
                                case 'image/png':
                                    shell_exec('convert '.$chemin_source.' '.$chemin_source.'.pdf');
                                    break;

                                default:
                                    break;
                            }

                            // Is conversion success?
                            if (file_exists($chemin_source.'.pdf')) {
                                $piece_justificative_pdf_files[] = $chemin_source.'.pdf';
                            }else{
                                $GLOBALS['log']->fatal("Erreur a la conversion de l'image en PDF ".$file_name);
                            }

                            // Remove the image(s) from 'upload_dir' used for conversion.
                            if (file_exists($chemin_source)) { unlink( $chemin_source ); }      
                            
                        }

                    }else{ // Erreur à la sauvegarde de la pj 
                        return false; 
                    }
                }

                $final_pdf_list  = join( " ",$piece_justificative_pdf_files );
                $tmp_file_chemin = $sugar_config['upload_dir'].$uuid .'_'. date('YmdHis').".pdf";

                $shell_exec_cmd  ="gs -dPDFA=1 -dBATCH -dNOPAUSE -dUseCIEColor -sProcessColorModel=DeviceCMYK -sDEVICE=pdfwrite -sPDFACompatibilityPolicy=1  -r300 -dDetectDuplicateImages -sOutputFile=".$tmp_file_chemin." ".$final_pdf_list;

                $shell_exec_result = shell_exec($shell_exec_cmd);

                $pdf_content = '';
                if (file_exists($tmp_file_chemin)) {

                    $pdf_content = file_get_contents($tmp_file_chemin);
                    $pdf_size = strlen($pdf_content);
                    unlink($tmp_file_chemin);


                }
                else{
                    $GLOBALS['log']->fatal(" OPS_dossier :: addJustificatif => Erreur à l'assemblage du multifichier" );

                }

                if( isset($champs_dossier['custom'][$key]['files']) ){
                    unset($champs_dossier['custom'][$key]['files']);
                }
                $objTypeDocument = BeanFactory::getBean('OPS_type_document', $value['attache']);

                $value['name'] = str_replace(' ', '_', $objTypeDocument->name) . '.pdf'; // Création du nom final basé sur le nom du type de document.
                $value['size'] = $pdf_size;
                $value['content'] = base64_encode($pdf_content);
                $value['type'] = 'application/pdf';
            }


            # On tente de détecter si le content est un chemin d'une PJ existante ou un Base64 brut.
            $flag_reuse_pj = false;
            if( is_array($value) && isset($value['content']) )
            {

                if( preg_match_all("/[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}/si", $value['content'], $matches) )
                {

                    // Construction correcte et concrète d'un chemin de PJ ?
                    if( isset($sugar_config['ops_justificatif']) && is_array($sugar_config['ops_justificatif'])
                     && isset($sugar_config['ops_justificatif']['chemin']) && !empty($sugar_config['ops_justificatif']['chemin'])
                     && sizeof($matches[0]) == 2 /* UUID Dossier/PersonneMoraleID  & UUID DocPJ */
                     && $value['content'] == $sugar_config['ops_justificatif']['chemin']."/".$matches[0][0].'/'.$matches[0][1] )
                    {
                        // Ceci est un chemin d'un PJ existante - cas réutilisation ou pré-enregistrée
                        $flag_reuse_pj = true;
                    }

                }
            }

            # On teste si la valeur de ce champ est bien un type champ avec ses données de base.
            if( is_array($value) 
             && isset($value['name'])    && !empty($value['name'])
             && isset($value['size'])    && !empty($value['size'])
             && isset($value['content']) && !empty($value['content'])
             && isset($value['type'])    && !empty($value['type'])
             && isset($value['attache']) && !empty($value['attache'])
            # /!\ Failed here
             /*
             && (empty($obj_dossier->ops_personne_morale) || empty(stristr($value['content'],$obj_dossier->ops_personne_morale))) 
             && empty(stristr($value['content'],$obj_dossier->id)) /**/ )
            {

                $ext = explode(".", $value['name']);

                # Cas où ce document PJ est un fichier importé (avec assemblage ou non)
                if( !$flag_reuse_pj ){

                    $objTypeDocument = BeanFactory::getBean('OPS_type_document', $value['attache']);

                    if ($objTypeDocument->objet == 'tiers' && empty($obj_dossier->ops_personne_morale)) {
                        // Réaffecte l'objet au dossier afin de ne pas perdre le justificatif.
                        $objTypeDocument->objet = 'dossier';
                    }

                    if( isset( $customChamp ) && $customChamp->{$key}->content != $value['content']) 
                    {
                        $idJustificatif = end(explode('/', $customChamp->{$key}->content));

                        if(!empty($idJustificatif))
                        {
                            $objJustificatif = BeanFactory::getBean('OPS_justificatif', $idJustificatif);
                            if(!empty($objJustificatif->id))
                            {
                                $objJustificatif->mark_deleted($objJustificatif->id);
                            }
                        }
                    }

                    $objJustificatif = BeanFactory::newBean('OPS_justificatif');
                    //if ($value['type'] == "application/pdf" || $value['type'] == "text/csv") {
                        $objJustificatif->document_name = $value['name'];
                        $objJustificatif->filename = $value['name'];
                        $objJustificatif->file_ext = end($ext);
                        $objJustificatif->file_mime_type = $value['type'];
                        $objJustificatif->visible_internet = $objTypeDocument->visible_internet;
                        $objJustificatif->visible_partenaire = $objTypeDocument->visible_partenaire;
                   /* } else { // On va forcer le passage en PDF
                        $value['name'] = explode('.', $value['name'])[0] . '.pdf'; 
                        $objJustificatif->document_name = $value['name'];
                        $objJustificatif->filename = $value['name'];
                        $objJustificatif->file_ext = 'pdf';
                        $objJustificatif->file_mime_type = 'application/pdf';
                    }*/

                    if(!empty($obj_dossier->ops_dispositif_id))
                    {
                        $dispositifModel = new DispositifModel($obj_dossier->ops_dispositif_id);
                        $champs_vue_agent = $dispositifModel->getChampsVueUsager();
                        if(!empty($champs_vue_agent) && 
                            is_array($champs_vue_agent) && 
                            array_key_exists($key, $champs_vue_agent) && 
                            array_key_exists('obligatoire', $champs_vue_agent[$key]))
                        {
                            $objJustificatif->obligatoire = $champs_vue_agent[$key]['obligatoire'];
                        }
                    }


                    $objJustificatif->save();

                    $objJustificatif->load_relationship('ops_type_document_ops_justificatif');
                    $objJustificatif->ops_type_document_ops_justificatif->add($objTypeDocument);

                    // Si tiers -> alors on associe le justificatif à la personne morale...
                    if ($objTypeDocument->objet == 'tiers') {
                        if(!empty($obj_personne_morale->id))
                        {
                            $obj_personne_morale->ops_personne_morale_ops_justificatif->add($objJustificatif);
                            $chemin = $sugar_config['ops_justificatif']['chemin'] . "/" . $obj_personne_morale->id;
                        }
                    }
                    // Sinon on associe le justificatif au dossier...
                    else {
                        $obj_dossier->ops_dossier_ops_justificatif->add($objJustificatif);
                        $chemin = $sugar_config['ops_justificatif']['chemin'] . "/" . $obj_dossier->id;
                    }

                    # Tentative de créer le chemin de destination qui accueillira la PJ.
                    if(!empty($chemin))
                    {
                        @mkdir($chemin, 0755);
                        if (is_dir($chemin)) {
    
                            $chemin_file_source = $chemin . '/' . $objJustificatif->id ; 
    
                            file_put_contents( $chemin_file_source , base64_decode( $value['content'] ) );
    
                            $obj_scan = new ScanPJ();
                            $chemin_verif = $_SERVER['DOCUMENT_ROOT'] .'/'. $chemin . '/' . $objJustificatif->id ; 
                            $ret = $obj_scan->scanPathForMalware( $chemin_verif );
    
                            
                        }
                    }


                    // On convertit tout en pdf avant de convertir en PDFA 
                /*    switch ( strtolower($value['type']) ) {
                        case 'image/jpeg':
                            shell_exec('convert '.$chemin_file_source.' '.$chemin_file_source.'.png');
                            shell_exec('convert '.$chemin_file_source.'.png '.$chemin_file_source.'.pdf'); 
                            if (file_exists($chemin_file_source.'.png')) { unlink( $chemin_file_source.'.png' ); }      
                            break;
                        case 'image/png':
                            shell_exec('convert '.$chemin_file_source.' '.$chemin_file_source.'.pdf');
                            break;

                        default:
                            break;
                    }


                    $shell_exec_cmd = "gs -dPDFA=1 -dBATCH -dNOPAUSE -dUseCIEColor -sDEVICE=pdfwrite -sPDFACompatibilityPolicy=1  -r300 -dDetectDuplicateImages -sOutputFile=" .$chemin_file_source . " " .$chemin_file_source.".pdf";                            
                            $shell_exec_result = shell_exec($shell_exec_cmd);
                    */

                } // Fin de cas d'un fichier importé.


                
                # On associe le document PJ (Créé) ou la PJ réutilisée (déjà existante) au champ afin d'en prendre compte dans le retour de peuplement des champs du dossier.
                if( $flag_reuse_pj ){
                    $chemin_pj = $value['content'];
                }else{
                    $chemin_pj = $chemin_file_source;

                    if($objTypeDocument->objet == 'dossier')
                    {
                        $champs_dossier['flag']['justificatif'] = 1;
                    }
                }

                $champs_dossier['custom'][$key] = []; // Reinit le champ sous forme de tableau vide pour peuplement correct des data.


                # Affectation des information de la PJ.
                if (!empty($chemin_pj) && is_file($chemin_pj)) {
                    $champs_dossier['custom'][$key]['content'] = $chemin_pj; // inclus le chemin de la ressource dans le content (inexploité après coup)
                    $champs_dossier['custom'][$key]['name']    = $value['name'];
                    $champs_dossier['custom'][$key]['size']    = $value['size'];
                    $champs_dossier['custom'][$key]['type']    = $value['type'];
                    $champs_dossier['custom'][$key]['attache'] = $value['attache'];
                } else {
                    $champs_dossier['custom'][$key]['content'] = 'file-error';
                    $champs_dossier['custom'][$key]['name']    = 'unknow';
                    $champs_dossier['custom'][$key]['size']    = '0';
                    $champs_dossier['custom'][$key]['type']    = 'unknow';
                    $champs_dossier['custom'][$key]['attache'] = '';
                }


            }

        }
        if( isset($piece_justificative_pdf_files) ){
            // Cleanup the store generated (temp) PDF.
            foreach ( $piece_justificative_pdf_files as $key => $tmp_file ) {
                unlink( $tmp_file );
            }
        }
        return $champs_dossier;
    }


    /**
     * @access private
     * @name createBeanDossier()
     * Fonction qui crée l'objet dossier
     *
     *  @param array            $onglets : la liste des onglets rattaché à la vue
     *  @return string        $btn_onglets_html : code html , les boutons sont groupés dans une div
     */
    private static function createBeanDossier($champs_dossier)
    {

        $champs_exclus = ["modified_user_id", "created_by", "assigned_user_id"];

        if (empty($champs_dossier['basic'])) {
            return false;
        }

        $obj_dossier = BeanFactory::newBean('OPS_dossier');
        $id = $obj_dossier->save(false);

        // On parcours les champs basic (disponibles via le formulaire -alias Vue-)...
        foreach ($champs_dossier['basic'] as $champ_name => $champ_value) {

            // On exclus les champs de base qui ont des valeurs par defaut et les valeurs vides
            if (!in_array($champ_name, $champs_exclus) && !empty($champ_value)) {
                if (!is_array($champ_value)) {
                    if ($obj_dossier->field_name_map[$champ_name]['type'] == 'enum' && substr_count($champ_value, '^') == 2) {
                        $champ_value = str_replace("^", "", $champ_value);
                    }
                    // Le champ n'est pas de type relation => insertion direct
                    $obj_dossier->$champ_name = $champ_value;
                } else if (!empty($champ_value['type']) && $champ_value['type'] == "parent_id") {
                    // Le champ est de type relation mais avec un parent_id => insertion direct
                    $obj_dossier->$champ_name = $champ_value['module_id'];
                } else {
                    // Le champ est de type relation mais avec une liaison par table => création de la relation
                    $module_ids = array();
                    if ($champ_value['unique'] === "multi" && strpos($champ_value['module_id'], '|') !== false) {
                        $champ_module_ids = explode("|", $champ_value['module_id']);
                        foreach ($champ_module_ids as $champ_module_id) {
                            if (!empty($champ_module_id)) {
                                $module_ids[] = $champ_module_id;
                            }
                        }
                    } else {
                        $module_ids[] = $champ_value['module_id'];
                    }

                    foreach ($module_ids as $module_id) {
                        $obj = BeanFactory::getBean($champ_value['module_name'], $module_id);
                        if (!empty($obj->id) && $champ_value['module_name'] != 'OPS_dossier') {
                            $obj->load_relationship($champ_value['relation_name']);
                            $obj->{$champ_value['relation_name']}->add($id);
                        } else if ($champ_value['module_name'] == 'OPS_dossier') {
                            $obj->load_relationship($champ_value['relation_name']);
                            $obj->{$champ_value['relation_name']}->add($obj_dossier->id);
                        }

                    }
                }
            }
        }

        //On ajoute les justificatfs
        $champs_dossier = OPS_dossier::addJustificatif($obj_dossier, $champs_dossier);

        if(isset($champs_dossier['flag']['justificatif']) && !empty($champs_dossier['flag']['justificatif']))
        {
            $obj_dossier->flag_justificatif_etude = $champs_dossier['flag']['justificatif'];
        }

        //On exécute les champs fonction
        $champs_dossier = OPS_dossier::executeFonctions($obj_dossier, $champs_dossier);
        $champs_dossier = OPS_dossier::executeGrille($obj_dossier, $champs_dossier);

        // On récpère les champs déjà existant avec leur valeurs, qu'on va merger avec les nouvelles valeurs transmises.
        $old_champs_custom = (isset($obj_dossier->champs_custom) && !empty($obj_dossier->champs_custom)) ? json_decode(base64_decode($obj_dossier->champs_custom), true) : array();
        $new_champs_custom = (!empty($champs_dossier['custom'])) ? $champs_dossier['custom'] : array();

        // Tous les champs custom du formulaire de la vue doivent être soit fournis soit remplis avec les valeurs existantes.
        $champs_custom = array_merge($old_champs_custom, $new_champs_custom);
        $obj_dossier->champs_custom = base64_encode(json_encode($champs_custom));

        $obj_dossier->ops_individu_id = $champs_dossier['extended']["demandeur"];

        $obj_dispositif = BeanFactory::getBean("OPS_dispositif", $champs_dossier['extended']["dispositif"]);
        if (!empty($obj_dispositif->id)) {
            $obj_dispositif->load_relationship("ops_dispositif_ops_dossier");
            $obj_dispositif->ops_dispositif_ops_dossier->add($id);
        }

        // Initialisation du numéro du dossier
        $obj_dossier = OPS_dossier::initNumAndNameDossier($obj_dossier);

        $obj_dossier->save();

        // Initialisation du statut et de l'étape du dossier
        $obj_dossier = OPS_dossier::initStatutEtapeDossier($obj_dossier, $champs_dossier['extended']['dispositif']);

        // On enregistre une fois l'objet dossier alimenté et les traitements annexes effectué avec succés.
        return ($obj_dossier->save()) ? $id : false;
    }

    /**
     * @access private
     * @name initStatutEtapeDossier()
     * Fonction qui initialise l'étape et le statut du dossier
     *
     *  @param bean          $obj_dossier : objet OPS_dossier
     *  @param array            $dossier : tableau des données du dossier
     *  @return bean            $obj_dossier : objet OPS_dossier
     */
    public static function initStatutEtapeDossier($obj_dossier, $dispositif_id)
    {

        // On récupere le dispositif
        $obj_dispositif = (!empty($dispositif_id)) ? BeanFactory::getBean('OPS_dispositif', $dispositif_id) : false;

        // On récupere le guide d'instruction associé au dispositif
        $liste_guide_instruction = (!empty($obj_dispositif->id)) ? $obj_dispositif->get_linked_beans('ops_guide_instruction_ops_dispositif', 'OPS_guide_instruction') : false;
        $guide_instruction = (is_array($liste_guide_instruction) && count($liste_guide_instruction) == 1) ? $liste_guide_instruction[0] : false;

        // On récupére le statut de démarrage du guide d'instruction
        $guide_instruction_statut_initial = ($guide_instruction !== false && !empty($guide_instruction->ops_statut_id)) ? $guide_instruction->ops_statut_id : false;

        // On initialise le premier statut du dossier (GI)
        $obj_dossier = OPS_dossier::setFirstStatut($obj_dossier, $guide_instruction_statut_initial);
        return $obj_dossier;
    }


    /**
     * @access private
     * @name initNumAndNameDossier()
     * Fonction qui initialise le numéro du dossier ( champ : num_dossier )
     *
     *  @param bean          $obj_dossier : objet OPS_dossier
     *  @param array            $dossier : tableau des données du dossier
     *  @return bean            $obj_dossier : objet OPS_dossier
     */
    private static function initNumAndNameDossier($obj_dossier)
    {
        // On récupere le numéro du dernier dossier
        $obj_dossier->num_dossier = DossierModel::getNumDossier($obj_dossier->id);

        // On initialise le nom du dossier
        $obj_dossier->name = (!empty($obj_dossier->libelle_dossier)) ? $obj_dossier->num_dossier . " - " . $obj_dossier->libelle_dossier : (string) $obj_dossier->num_dossier;

        return $obj_dossier;
    }

    /**
     * @access private
     * @name getDataDossier()
     * Fonction qui génére les boutons d'onglets
     *
     *  @param array            $onglets : la liste des onglets rattaché à la vue
     *  @return string        $btn_onglets_html : code html , les boutons sont groupés dans une div
     */
    private static function getDataDossier($champs_formulaire, $champs_onglets)
    {

        $erreurs = array();
        $dossier = array();
        $demandeur_type = $champs_formulaire["type_tiers"];
        // On parcours les champs de la vue dossier
        foreach ($champs_onglets as $champ_onglet_id => $champ_onglet) {
            // Si le champ existe dans les champs du formulaire
            if (array_key_exists($champ_onglet_id, $champs_formulaire)) {
                // Si le champ du formulaire n'est pas vide
                if (!empty($champs_formulaire[$champ_onglet_id]) || $champs_formulaire[$champ_onglet_id] === "0") {
                    // On récupere la valeur du champ
                    $value = OPS_dossier::getChampValueByType($champs_formulaire[$champ_onglet_id], $champ_onglet, $demandeur_type);
                    if ($value !== false) {
                        $dossier[$champ_onglet['description']][$champ_onglet_id] = $value;
                    } else {
                        if ($champs_formulaire['canal'] != "internet") { // Pas de gestion des erreurs au niveau du front
                            $erreurs[] = "Echec de la récupération de la valeur du champ " . $champ_onglet['libelle'];
                        }
                    }
                } else {
                    // Si le champ est obligatoire et vide, on ajoute l'erreur au tableau d'erreurs
                    if ($champ_onglet['obligatoire'] != 1) {
                        $dossier[$champ_onglet['description']][$champ_onglet_id] = OPS_dossier::getChampVidebyType($champ_onglet, $demandeur_type);
                    } else {
                        /*if ($champs_formulaire['canal'] != "internet") { // Pas de gestion des erreurs au niveau du front
                            $erreurs[] = "Le champ " . $champ_onglet['libelle'] . " est obligatoire.";
                        }*/
                    }
                }
            }
        }

        return (is_array($erreurs) && count($erreurs) > 0) ? $erreurs : $dossier;
    }

    /*******************************************************************************************************************************************/
    /********************************************************* Gestion des habilitations ********************************************************/
    /*******************************************************************************************************************************************/


    // Ajout des habilitations sur les vues listes DOSSIERS
    function create_new_list_query($order_by, $where, $filter = array(), $params = array(), $show_deleted = 0, $join_type = '', $return_array = false, $parentbean = null, $singleSelect = false, $ifListForExport = false)
    {
        global $current_user;
        $jsonRequest = $_REQUEST['json'] ?? false;



        $fromMassUpdate = false;

        if($jsonRequest != false){
            $decodedJson = json_decode(base64_decode($_REQUEST['json']), true);
            $gestion_dossier_mu = $decodedJson['filtres']['gestion_dossier_mu'] ?? false;
            $gestion_dossier_mu = json_decode(base64_decode($gestion_dossier_mu),true);
            $fromMassUpdate = json_last_error() === JSON_ERROR_NONE;
        }


        $casGestionDossier = ((isset($_REQUEST['gestion_dossier']) || isset($_GET["gestion_dossier"])) && isset($_GET['GestionView'])) || $fromMassUpdate;

        $return_array = parent::create_new_list_query($order_by, $where, $filter, $params, $show_deleted, $join_type, $return_array, $parentbean, true, $ifListForExport);

        $requeteAEteTransforme = false;
        if($casGestionDossier){

            $gestionDossier = $fromMassUpdate ? $gestion_dossier_mu : OPS_dossier::getParamRequestToArray('gestion_dossier', "REQUEST") ;

            if(empty($gestionDossier) && !empty($_GET["gestion_dossier"])) 
                $gestionDossier = $_GET["gestion_dossier"];
            else
                $_GET["gestion_dossier"] = $gestionDossier;


            if(!is_array($return_array)){
                $requeteAEteTransforme = true;
                $return_array = OdeStringHelper::parseSQL($return_array);
            }


            foreach ($gestionDossier as $type => $filtres) {
                foreach ($filtres as $champ => $infos) {
    
                    // Préfixe de table pour les champs
                    $prefixeTable = $infos['table'] . ($infos['table'] == "" ? "" : ".");
    

                    // Référence à la clause WHERE de la requête
                    $clauseWhere = $return_array["where"];
    
                    // Valeurs des filtres
                    $valeursFiltre = explode("|", $infos['valeurs']);
            
                    // Traitement des filtres simples
                    if ($type === "filtres_simples") {
                        $clauseWhere .= " AND $prefixeTable$champ " . (count($valeursFiltre) == 1 ? "= '{$valeursFiltre[0]}'" : 'IN ("' . implode('","', $valeursFiltre) . '")');
                    } 
                    // Traitement des filtres simples avec flag
                    elseif ($type === "filtres_simples_flag") {
                        // Construction de la condition WHERE avec des OR
                        $clauseWhere .= " AND (";
                        foreach (explode("|", $infos['colonnes']) as $colonne) {
                            $clauseWhere .= "$prefixeTable$colonne = {$infos['valeurs']} OR ";
                        }
                        // Suppression du dernier " OR "
                        $clauseWhere = rtrim($clauseWhere, " OR ") . ")";
                    }
                    elseif ($type === "filtres_simples_relation") {
                        $from = 
                        " LEFT JOIN {$infos['relate']} ON {$infos['relate']}.{$infos['champs1r']} = {$infos['table']}.{$infos['champ']}
                          INNER JOIN {$infos['table2']} ON {$infos['table2']}.{$infos['champ']} = {$infos['relate']}.{$infos['champs2r']}";
    
    
                        $return_array['from'] .= $from;
                        $clauseWhere .= " AND {$infos['table2']}.{$infos['champ']} " . (count($valeursFiltre) == 1 ? "= '{$valeursFiltre[0]}'" : 'IN ("' . implode('","', $valeursFiltre) . '")');
                        $clauseWhere .= " AND {$infos['relate']}.deleted = 0";

    
                    }
                    $return_array['where'] = $clauseWhere;

                    
                }
            }

            $return_array['where'] .= ' AND ops_dossier.name != "TEMP" ';

            $_GET['from'] = $return_array["from"];
            $_GET['where'] = $return_array["where"];

            if($requeteAEteTransforme)
                $return_array = $return_array['select'] . " " . $return_array['from'] . " " . $return_array['where'] . " " . $return_array['order_by'];


        }



        unset($_REQUEST['gestion_dossier']);

        if(!is_array($return_array))
        {
            return $return_array;
        }

        $getPreference = $current_user->getPreference("OPS_dossierQ");

        if ((!empty($_GET['filter']) &&  "nogps" == $_GET['filter'])  || ($getPreference['geoloc'] == "nogps" || $getPreference['filter'] == "nogps") && "" == $_GET['action']) 
        {
            $getPreference["geoloc"] = "npogps";
            unset($getPreference["filter"]);
            $current_user->setPreference("OPS_dossierQ", $getPreference);

            $getPreference = $current_user->getPreference("OPS_dossierQ");
            $return_array['where'] .= " AND (statut_geoloc = 'erreur' OR statut_geoloc = 'empty') ";
        } 
        else 
        {
            unset($getPreference["filter"]);
            $current_user->setPreference("OPS_dossierQ", $getPreference);
        }

        // On récupère les dispositifs 
        $this->create_requete_dossier();

        // Filtrage des dossiers temporaires
        $return_array['where'] .= ' AND ops_dossier.name != "TEMP" ';

        if(!$current_user->isAdmin() && $_SESSION['ops_requete_dossier'] == false)
        {
            $return_array['where'] = "  WHERE ops_dossier.id like '0' ";

            return $return_array;
        }

        # Problème de perfs pour le boutton "sélectionner" d'un sous-pannel
        if($_REQUEST['action'] == 'Popup')
        {
            $relations = ['ops_reunion','ops_commission','ops_sous_territoire'];
            $return_array = $this->cleanQuery($return_array, $relations);
        }


        //$GLOBALS['log']->fatal(print_r($return_array,true));

        //$return_array['where'] .= " AND ops_etape_id = 'f1f95b5f-217e-2cf4-3043-662a03ee946f'";
        if($current_user->isAdmin())
        {
            return $return_array;
        }

        // ON RETIRE DU SELECT LES FONCTIONS SQL ET LES VALEURS PAR DEFAUT
        $return_array['select'] = str_replace(
            "LTRIM(RTRIM(CONCAT(IFNULL(jt1.first_name,''),' ',IFNULL(jt1.last_name,'')))) ops_individu_ops_dossier_name",
            "CONCAT(jt1.first_name, ' ',jt1.last_name) ops_individu_ops_dossier_name",
            $return_array['select']
        );
        $return_array['select'] = str_replace(
            "LTRIM(RTRIM(CONCAT(IFNULL(jt3.first_name,''),' ',IFNULL(jt3.last_name,'')))) beneficiaire",
            "CONCAT(jt3.first_name, ' ',jt3.last_name) beneficiaire",
            $return_array['select']
        );

        $return_array['select'] = str_replace(
            "'OPS_statut' statut_mod,",
            '',
            $return_array['select']
        );

        $return_array['select'] = str_replace(
            "'Users' assigned_user_name_mod,",
            '',
            $return_array['select']
        );

        $return_array['select'] = str_replace(
            "'OPS_etape' etape_mod ,",
            '',
            $return_array['select']
        );
        // FIN PARTIE SELECT

        // DISPOSITIFS
        $return_array['where'] .= ' AND ( ' . $_SESSION['ops_requete_dossier'];

        // DOSSIER LOCALISE OU NON
        if( !$current_user->isAdmin() && (!empty($_SESSION['habilitation_localisation']) || $current_user->dossiers_non_localises == true) )
        {
            $from = $where_localisation = $where_non_localise = '';

            if( !empty($_SESSION['habilitation_localisation']) )
            {
                $user_localisations = $_SESSION['habilitation_localisation'];
                $localisations = [];
                // Partie TERRITOIRE
                foreach($user_localisations as $localisation)
                {
                    $localisations[] = "'" . $localisation . "'";
                }

                $localisations = implode(',', $localisations);

                $where_localisation = " ops_dossier.id IN ( SELECT ops_dossier.id
                                                            FROM ops_dossier 
                                                            INNER JOIN ops_sous_territoire_ops_dossier sous_territoire_dossier ON ops_dossier.id = sous_territoire_dossier.ops_dossier_id AND sous_territoire_dossier.deleted=0
                                                            INNER JOIN ops_sous_territoire_ops_territoire sous_territoire_territoire ON sous_territoire_dossier.ops_sous_territoire_id = sous_territoire_territoire.ops_sous_territoire_id AND sous_territoire_territoire.deleted=0
                                                            WHERE ops_dossier.deleted = 0 AND sous_territoire_territoire.ops_territoire_id IN ( $localisations ) )";
            }

            $return_array['where'] .= ' ) ';

            if( $current_user->dossiers_non_localises == true )
            {
                $where_non_localise = "  ops_dossier.id NOT IN ( 
                                             SELECT  ops_sous_territoire_ops_dossier.ops_dossier_id
                                             FROM ops_sous_territoire_ops_dossier
                                             WHERE ops_sous_territoire_ops_dossier.deleted = '0'
                                         )";
            }

            if( empty($where_localisation) && !empty($where_non_localise))
            {
                $return_array['where'] .= ' AND (' . $where_non_localise . ')';
            }
            if( !empty($_SESSION['habilitation_localisation']) && !empty($where_localisation) && $current_user->dossiers_non_localises == true )
            {
                $return_array['where'] .= ' AND (' . $where_localisation . ' OR ' . $where_non_localise . ') ';
            }
            else
            {
                $return_array['where'] .= ' AND ' . ((!empty($where_localisation)) ? $where_localisation : $where_non_localise);
            }

            $return_array['from'] .= $from;
        }

        // END - DOSSIER LOCALISE OU NON

        $return_array['from_min'] = $return_array['from'];

        // ORDER BY
        if (is_array($return_array) && (empty($return_array['order_by']) || $return_array['order_by'] == " ORDER BY ops_dossier.num_dossier")) 
        {
            $return_array['order_by'] = " ORDER BY " . strtoupper('num_dossier') . ' DESC';
        }

        return $return_array;
    }


    public static function getParamRequestToArray($nomParam, $tableau="GET"){
        $source = $_GET;
        if($tableau == "REQUEST") $source=$_REQUEST;

        $param = $source[$nomParam] ?? array();
        $param = !is_array($param) ? htmlspecialchars_decode($param) : $param;
        $param = !is_array($param) ? json_decode($param, true) : $param;
        return $param;
    }
    private function cleanQuery($return_array, $relations)
    {
        foreach ($relations as $relation) 
        {
            if (preg_match('/'.$relation.' jt(\d+)/', $return_array['from'], $matches)) 
            {
                $numero = $matches[1];

                if(strpos($return_array['from'],'jt'.$numero) !== false && strpos($return_array['from'],'jtl'.$numero) !== false)
                {
                    $return_array['from'] = str_replace(
                        'LEFT JOIN  '.$relation.' jt' . $numero . ' ON jt' . $numero . '.id=jtl' . $numero . '.'.$relation.'_id',
                        '',
                        $return_array['from']
                    );

                    $return_array['from'] = str_replace(
                        'AND jt' . $numero . '.deleted=0',
                        '',
                        $return_array['from']
                    );

                    $return_array['from'] = str_replace(
                        'LEFT JOIN  '.$relation.'_ops_dossier jtl' . $numero . ' ON ops_dossier.id=jtl' . $numero . '.ops_dossier_id AND jtl' . $numero . '.deleted=0',
                        '',
                        $return_array['from']
                    );

                    if (preg_match("/jt" . $numero . "(.*?)(,|FROM)/", $return_array['select'], $matches)) 
                    {
                        $result = 'jt' . $numero . $matches[1];

                        if(isset($matches[2]) && $matches[2] == ',')
                        {
                            $result .= $matches[2]; 
                        }

                        $return_array['select'] = str_replace(
                            $result,
                            '',
                            $return_array['select']
                        );
                    } 

                    if (preg_match("/jtl" . $numero . "(.*?)(,|FROM)/", $return_array['select'], $matches)) 
                    {
                        $result = 'jtl' . $numero . $matches[1];

                        if(isset($matches[2]) && $matches[2] == ',')
                        {
                            $result .= $matches[2]; 
                        }

                        $return_array['select'] = str_replace(
                            $result,
                            '',
                            $return_array['select']
                        );
                    } 
                }
            }
        }
        return $return_array;
    }



    /**
     * create_requete_dossier
     * Fonction du hook permettant les habilitations des localisations sur les utilisateurs
     *
     * @param  mixed $bean
     * @param  mixed $event
     * @param  mixed $arguments
     *
     * @return void
     */
    function create_requete_dossier()
    {
        global $current_user;

        // Préparation de la requete 
        $tab_localisation = $_SESSION['habilitation_localisation'];
        $nb_loca = (is_array($tab_localisation)) ? count($tab_localisation) : 0;

        $tab_dispositif = $_SESSION['habilitation_dispositif'];
        $nb_dispo = (is_array($tab_dispositif)) ? count($tab_dispositif) : 0;

        $ids_dispo = array();
        $ids_etapes = array();

        $where_dispo_dossier = '';

        //Création du where des dispositifs 
        if ($nb_dispo > 0) 
        {
            foreach ($tab_dispositif as $id => $dispo) 
            {
                // Vérification que l'utilisateur connecté est le droit de visualisation du dispositif 
                if (!empty($id) && $dispo['visualisation'] == 1) 
                {
                    $ids_etape = array();

                    // gestion du dispositif
                    $ids_dispo[] = "'" . $id . "'";

                    // gestion des étapes 
                    if (array_key_exists("etapes", $dispo)) 
                    {
                        // Récupération des droits étapes 
                        foreach ($dispo['etapes'] as $id_etape => $etape) 
                        {
                            if ($etape['visualisation'] == 1) 
                            {
                                $ids_etapes[] = "'" . $id_etape . "'";
                                $ids_etape[] = "'" . $id_etape . "'";

                                // Si l'étape correspond à un dispositif 
                                if (!in_array($id, $ids_dispo)) 
                                {
                                    $ids_dispo[] = "'" . $id . "'";
                                }
                            }
                        }
                    }

                    if(!empty($where_dispo_dossier))
                    {
                        $where_dispo_dossier .= ' OR ';
                    }

                    $where_dispo_dossier .= " ops_dossier.id in ( select ops_dossier.id from ops_dossier LEFT JOIN ops_dispositif_ops_dossier ON ops_dispositif_ops_dossier.ops_dossier_id=ops_dossier.id AND ops_dispositif_ops_dossier.deleted=0 where  ops_dispositif_ops_dossier.ops_dispositif_id = '" . $id . "' ";


                    if (!empty($ids_etape)) 
                    {
                        $ids_etape = implode(",", array_unique($ids_etape));
                        $where_dispo_dossier .= " AND ( ops_dossier.ops_etape_id IS NULL OR ops_dossier.ops_etape_id in ('', " . $ids_etape . " ) ) ";
                    }

                    $where_dispo_dossier .=  ")";
                }
            }

            if (!empty($ids_dispo)) 
            {
                $where_dispo = " ops_dispositif.id in (" . implode(',', $ids_dispo) . " ) ";
            }
            else 
            {
                $where_dispo = " ops_dispositif.id = '0' ";
            }
        }

        $isHabilite = $where_dispo_dossier;

        // Si l'utilisateur n'a pas de territoire et qu'il n'a pas la case Dossiers non localisés coché alors il ne voit aucun dossier 
        if ($nb_loca == 0 &&  $current_user->dossiers_non_localises == false) 
        {
            $isHabilite = false;
        }

        $_SESSION['ops_requete_dossier'] = $isHabilite;
        $_SESSION['ops_requete_dispositif'] = $where_dispo;
    }

    public static function duplicateDossier($dossier_id){

        $DossierModel = new DossierModel($dossier_id);
        $dossier_champs = $DossierModel->getChamps();
        $dossier = $DossierModel->getBean();

        if (is_array($dossier_champs) && count($dossier_champs) > 0) {

            // On crée le nouveau dossier
            $obj_dossier = BeanFactory::newBean('OPS_dossier');
            foreach ($dossier_champs as $champ_name) {
                $obj_dossier->$champ_name = $dossier[$champ_name];
            }
            $obj_dossier->doublon = 'oui';
            $obj_dossier->brouillon = 'oui';
            $obj_dossier->duplication_id = $dossier['id'];

            $obj_dossier_id = $obj_dossier->save(false);

            $modules = [
                [
                    "ids" => [$DossierModel->getDispositifId()],
                    "name" => "OPS_dispositif",
                    "relation_name" => "ops_dispositif_ops_dossier",
                ],
                [
                    "ids" => [$DossierModel->getIndividuId()],
                    "name" => "OPS_individu",
                    "relation_name" => "ops_individu_ops_dossier",
                ],
                [
                    "ids" => $DossierModel->getSousTerritoireIds(),
                    "name" => "OPS_sous_territoire",
                    "relation_name" => "ops_sous_territoire_ops_dossier",
                ],
            ];

            foreach ($modules as $module) {
                if (is_array($module["ids"]) && count($module["ids"]) > 0) {
                    foreach ($module["ids"] as $module_id) {
                        if (!empty($module_id)) {
                            $obj = BeanFactory::getBean($module['name'], $module_id);
                            if (!empty($obj->id)) {
                                $obj->load_relationship($module['relation_name']);
                                $obj->{$module['relation_name']}->add($obj_dossier_id);
                            }
                        }
                    }
                }
            }

            $obj_dossier->champs_custom = OPS_dossier::duplicateJustificatif($dossier, $dossier['champs_custom'], $obj_dossier_id);
            $obj_dossier = OPS_dossier::initNumAndNameDossier($obj_dossier);

            $obj_dossier_id = ($obj_dossier->save()) ? $obj_dossier_id : false;

            return $obj_dossier_id;
        }
    }

    public static function duplicateJustificatif($dossier,$champs_custom,$obj_dossier_id){

        global $sugar_config;

        $dossierOld = BeanFactory::getBean('OPS_dossier', $dossier['id']);
        $obj_dossier = BeanFactory::getBean('OPS_dossier', $obj_dossier_id);
        $champsCustom = json_decode(base64_decode($champs_custom), true);
        $justificatifs = '';

        foreach ($champsCustom as $key => $value) {
            if(is_array($value) 
                && isset($value['name']) && !empty($value['name'])
                && isset($value['size']) && !empty($value['size'])
                && isset($value['content']) && !empty($value['content'])
                && isset($value['type']) && !empty($value['type'])
                && isset($value['attache']) && !empty($value['attache'])
                    && empty(stristr($value['content'],$dossier['ops_personne_morale'])) 
                    && !empty(stristr($value['content'],$dossier['id']))){

                $justificatif_id = explode('/', $value['content']);
                $justificatif_id = $justificatif_id[count($justificatif_id) - 1];
                $ext = explode(".", $value['name']);
                $objTypeDocument = BeanFactory::getBean('OPS_type_document', $value['attache']);

                $justificatifOld = BeanFactory::getBean('OPS_justificatif', $justificatif_id);
                $objJustificatif = BeanFactory::newBean('OPS_justificatif');

                $justificatif_champs = OPS_dossier::getChampsJustificatif($justificatif_id);
                foreach ($justificatif_champs as $justificatif_name) {
                    $objJustificatif->$justificatif_name = $justificatifOld->{$justificatif_name};
                }
                $objJustificatif->save();

                $objJustificatif->load_relationship('ops_type_document_ops_justificatif');
                $objJustificatif->ops_type_document_ops_justificatif->add($objTypeDocument);

                $obj_dossier->load_relationship("ops_dossier_ops_justificatif");
                $obj_dossier->ops_dossier_ops_justificatif->add($objJustificatif);
                $chemin = $sugar_config['ops_justificatif']['chemin'] . "/" . $obj_dossier_id;

                @mkdir($chemin, 0755);
                if (is_dir($chemin)) {
                    copy($value['content'], $chemin . '/' . $objJustificatif->id);
                }

                // Reinit le champ sous forme de tableau vide pour peuplement correct des data.
                $champsCustom[$key] = [];

                if (is_file($chemin . '/' . $objJustificatif->id)) {
                    $champsCustom[$key]['content'] = $chemin . '/' . $objJustificatif->id; // inclus le chemin de la ressource dans le content (inexploité après coup)
                    $champsCustom[$key]['name'] = $value['name'];
                    $champsCustom[$key]['size'] = $value['size'];
                    $champsCustom[$key]['type'] = $value['type'];
                    $champsCustom[$key]['attache'] = $value['attache'];
                } else {
                    $champsCustom[$key]['content'] = 'file-error';
                    $champsCustom[$key]['name'] = 'unknow';
                    $champsCustom[$key]['size'] = '0';
                    $champsCustom[$key]['type'] = 'unknow';
                    $champsCustom[$key]['attache'] = '';
                }
                $justificatifs .= $justificatif_id . ';';
            }
        }

        $dossierOld->load_relationship("ops_dossier_ops_justificatif");
        $ancienJustificatif = $dossierOld->ops_dossier_ops_justificatif->get();

        foreach ($ancienJustificatif as $justificatif) {
            if (empty(stristr($justificatifs, $justificatif)) && !empty($justificatif)) {

                $justificatifOld = BeanFactory::getBean('OPS_justificatif', $justificatif);
                $justificatifOld->load_relationship('ops_type_document_ops_justificatif');
                $typeDocument = $justificatifOld->ops_type_document_ops_justificatif->get();

                $objTypeDocument = BeanFactory::getBean('OPS_type_document', $typeDocument[0]);

                $objJustificatif = BeanFactory::newBean('OPS_justificatif');

                $justificatif_champs = OPS_dossier::getChampsJustificatif($justificatif);

                foreach ($justificatif_champs as $justificatif_name) {
                    $objJustificatif->$justificatif_name = $justificatifOld->{$justificatif_name};
                }
                $objJustificatif->save();

                $objJustificatif->load_relationship('ops_type_document_ops_justificatif');
                $objJustificatif->ops_type_document_ops_justificatif->add($typeDocument[0]);

                $obj_dossier->load_relationship("ops_dossier_ops_justificatif");
                $obj_dossier->ops_dossier_ops_justificatif->add($objJustificatif);
                $chemin = $sugar_config['ops_justificatif']['chemin'] . "/" . $obj_dossier_id;

                if (!empty($justificatifOld->filename)) {
                    $cheminOld = $sugar_config['ops_justificatif']['chemin'] . "/" . $dossier['id'];
                    $chemin = $sugar_config['ops_justificatif']['chemin'] . "/" . $obj_dossier_id;
                    @mkdir($chemin, 0755);
                    if (is_dir($chemin)) {
                        copy($cheminOld . '/' . $justificatif, $chemin . '/' . $objJustificatif->id);
                    }
                }
            }
        }

        $champs_custom = base64_encode(json_encode($champsCustom));

        return $champs_custom;
    }

    public static function getChampsJustificatif($id)
    {
        global $db;

        $justificatifs = array();

        $sql = "SELECT * FROM `ops_justificatif` WHERE `id` = '" . $id . "' AND `deleted` = '0' ";
        $result = $db->query($sql);
        while ($row = $db->fetchByAssoc($result)) {
            $justificatifs[] = $row;
        }
        $justificatif = (!empty($justificatifs) && count($justificatifs) === 1) ? $justificatifs[0] : array();
        $champs = [];
        $champs_exclu = ["id", "name", "date_entered", "date_modified", "modified_user_id", "created_by", "deleted"];

        foreach ($justificatif as $champ_name => $champ_value) {
            if (!in_array($champ_name, $champs_exclu)) {
                $champs[] = $champ_name;
            }
        }

        return $champs;
    }

    /**
     * @access private
     * @name executeFonctions()
     * Fonction qui exécute les fonctions
     *
     *  @param array            $obj_dossier : dossier
     * @param array            $champs_dossier : Champs du dossier
     *  @return array        $champs_dossier : Champs du dossier
     */
    private static function executeFonctions($obj_dossier, $champs_dossier){

        if(empty($obj_dossier->ops_dispositif_id)){
            $obj_dossier->ops_dispositif_id = $champs_dossier['extended']['dispositif'];
        }

        if(!empty($obj_dossier->ops_dispositif_id)){
            $dispositifModel = new DispositifModel($obj_dossier->ops_dispositif_id);
            if($obj_dossier->canal == 'crm'){
                $champsVue = $dispositifModel->getChampsVueAgent();
            }else if($obj_dossier->canal == 'internet' || $obj_dossier->canal == 'hors_compte'){
                $champsVue = $dispositifModel->getChampsVueUsager();
            }
        }

        $fonctionNamespace = 'ODE\\Fonctions\\Liste\\';

        foreach ($champsVue as $key => $champ) {
            if($champ['type'] == "fonction" && !empty($champ['params'])){
                $params = json_decode(base64_decode($champ['params']),true);
                
                if(!empty($params['fonction'])){
                    $classeName = $fonctionNamespace . ucfirst(str_replace(".php", "", $params['fonction']));
                    if(class_exists($classeName)){
                        $classeObj = new $classeName();

                        $paramsFonction['dossier'] = $obj_dossier;
                        $paramsFonction['champs'] = $champs_dossier;

                        $champs_dossier[$champ['description']][$key] = $classeObj->execute($paramsFonction);
                    }
                }
            }
        }
        return $champs_dossier;
    }

    private static function executeGrille($obj_dossier, $champs_dossier){
        global $app_list_strings;
        
        // verification que le module tarification est installé
        if(!isset($app_list_strings['moduleList']['OPS_grille_tarifaire'])) return $champs_dossier;

        function nettoyerTexte($texte) {
            if(is_numeric($texte)) return $texte;
            // Convertir les entités HTML en caractères normaux
            $texteCopy = html_entity_decode($texte, ENT_QUOTES, 'UTF-8');
            // Supprimer tous les caractères non alphabétiques et les espaces
            $texteCopy = preg_replace('/[^a-zA-Z0-9\s]/', '', $texteCopy);
            $texteCopy = str_replace('^', '', $texteCopy);
            if($texteCopy == "") return $texte;
            return "'".$texteCopy."'";
        }

        if(empty($obj_dossier->ops_dispositif_id)){
            $obj_dossier->ops_dispositif_id = $champs_dossier['extended']['dispositif'];
        }

        if(!empty($obj_dossier->ops_dispositif_id)){
            $dispositifModel = new DispositifModel($obj_dossier->ops_dispositif_id);
            if($obj_dossier->canal == 'crm'){
                $champsVue = $dispositifModel->getChampsVueAgent();
            }else if($obj_dossier->canal == 'internet' || $obj_dossier->canal == 'hors_compte'){
                $champsVue = $dispositifModel->getChampsVueUsager();

                if(method_exists($dispositifModel, "getChampsVuePartenaire")){
                    $champsVuePartenaire = $dispositifModel->getChampsVuePartenaire();
                    $champsVue = array_merge($champsVuePartenaire, $champsVue);
                }
            }
        }


        $champs_dossier2 = array_merge($champs_dossier['basic'], $champs_dossier['custom']);

        // pour tous les champs grille
        foreach ($champsVue as $nom_technique => $champ) {



            // on continue si ce n'est pas une grille tarifaire et si les champs params sont vide
            if ($champ['type'] != "grille_tarifaire" || empty($champ['params'])) continue;

            // dans le cas ou on est sur le front on regarde qu'on l'ai dans l'onglet
            // ce qui implique que la grille tarifaire doit être dans le dernier onglet
            if(!isset($champs_dossier2[$nom_technique])) continue;


            $formuleValide = null;


            $journal = BeanFactory::newBean("OPS_journal");
            $journal->name = "Calcul du champ de type Grille tarifaire : " . $champ['name']; 
            $journal->parent_type = "OPS_dossier";
            $journal->parent_id = $obj_dossier->id;
            $journal->statut = 'ok';
            $journal->save();
            $journal->load_relationship('ops_journal_detail_ops_journal');



            //recuperation des param et de la grille
            $params = json_decode(base64_decode($champ['params']),true);
            
            $idGrille = $params["grille_tarifaire"];
            if(empty($idGrille)) continue;
            $beanGrille = BeanFactory::getBean("OPS_grille_tarifaire", $idGrille);
            if(empty($beanGrille->id)) continue;


            $donnesGrille = json_decode(base64_decode($beanGrille->donnees_grilles));

            // formules et vardef de la grille
            $formules = $donnesGrille->formules;
            $vardef = $donnesGrille->vardefs;

            $arrayJournalDetail = [];


            $champsCustomDossierSauvegarde = json_decode(base64_decode($obj_dossier->champs_custom),true) ?? [];

            // boucle sur les differents cas
            foreach($formules as $key => $formule){
                $journal_detail = BeanFactory::newBean("OPS_journal_detail");
                $journal_detail->name = "Vérification du cas - '" .$formule->nomDuCas ."'";
                $journal_detail->tache = "La valeur du cas '" . $formule->nomDuCas . "' sera elle utilisée pour le champ '" . $champ['name'] . "' ?"; 
                $journal_detail->ordre = $key + 1;


                $valeur = $formule->valeur;
                $toutesConditionsValides = true;

                // boucle sur les conditions
                foreach($formule->conditions as $num => $condition){

                    // Cette regex correspond à des variables, opérateurs, comparateurs, nombres et chaînes de caractères.
                    // variable : {var_test_x}
                    // operateurs/comparateurs : >= <= = > ...
                    $pattern = '/({[a-zA-Z_][a-zA-Z0-9_]*}|[+\-*\/()]|<=?|>=?|==?|!=|<|>|-??\d+(\.\d+)?|\'(?:[^\'\\\\]|\\\\.)*\')/';
                    preg_match_all($pattern, $condition, $formuleDecoupe);

                    // tableau de la formule avec chaque operande
                    $formuleDecoupe = $formuleDecoupe[0] ?: [];
                    $formuleVariableRemplace = "";

                    // boucle sur les operandes
                    foreach($formuleDecoupe as $operande){

                        if($operande == "=") $operande = "==";

                        // cas de variable
                        if(preg_match("/{[a-zA-Z_][a-zA-Z0-9_]*}/", $operande)){

                            // suppression des accolades
                            $variable = rtrim(ltrim($operande,"{"), "}");

                            // verification de la source de la variable 
                            $source = $vardef->{$variable}->source;
                            if($source == "OPS_dossier"){

                                
                                $valeur = $champs_dossier2[$variable];
                                // si on est sur le front notre variable de la condition n'est pas forcement dans l'onglet courant
                                // on regarde si c'est un basic du dossier
                                if(empty($valeur)) $valeur = $obj_dossier->{$variable} ?? "";
                                // et on regarde dans les custom
                                if(empty($valeur)) $valeur = $champsCustomDossierSauvegarde[$variable] ?? "";


                                $valeur = str_replace('^', '', $valeur ?? '');

                                // ajout des guillemets si on est sur un string
                                if(!is_numeric($valeur)) $valeur = "'" . strtolower($valeur) . "'";
                                $formuleVariableRemplace .= $valeur;

                            }
                            else if($source == "OPS_personne_morale"){
                                // recuperation du bean de la PM
                                $pm = BeanFactory::getBean($source,$obj_dossier->ops_personne_morale);
                                if($variable == "type_tier")
                                    $valeur = "'" . strtolower($pm->ops_type_personne_ops_personne_morale_name) . "'";
                                else 
                                    $valeur = "'" . strtolower($pm->{$variable}) . "'" ?? "";

                                $formuleVariableRemplace .= nettoyerTexte($valeur);

                            }
                            else if($source == "OPS_individu"){
                                $id = empty($obj_dossier->beneficiaire_id) ? $obj_dossier->ops_individu_id : $obj_dossier->beneficiaire_id;
                                $individu = BeanFactory::getBean($source,$id);
                                $valeur = "'" . strtolower($individu->{$variable}) . "'" ?? "";
                                $formuleVariableRemplace .= nettoyerTexte($valeur);

                            }
                            continue;
                        }
                        $formuleVariableRemplace .= nettoyerTexte(strtolower(strval($operande)));

                    } // fin boucle operande

                    $resultatCondition = "";
                    $journal_detail->description .= $num == 0 ? "" : "&nbsp;\n";
                    $journal_detail->description .= "#### Condition " . $num + 1 . " : \n";
                    $journal_detail->description .= "➤ Evaluation de la condition : " . $condition . "\n";
                    $journal_detail->description .= "➤ Condition avec les variables remplacé :  " . $formuleVariableRemplace . "\n";

                    // tentative d'evaluation de la variable
                    try {
                        eval( "\$resultatCondition = $formuleVariableRemplace;" );


                    } catch (ParseError $e) {
                        $toutesConditionsValides = false;
                        $journal_detail->description .= "➤ Erreur d'évaluation de la condition" . "\n";
                        $journal_detail->erreur .= "Erreur d'évaluation de la condition";
                        $GLOBALS['log']->fatal(print_r("OPS_dossier::executeGrille() => Erreur de formule : " . $formuleVariableRemplace,true));
                        break;
                    }
                    $journal_detail->description .= "➤ Résultat de la condition : " . ($resultatCondition ? "vraie" : "fausse") . "\n";
                    
                    $journal_detail->statut = "ok";

                    if(!$resultatCondition) {
                        $journal_detail->reponse = "Non";

                        $journal_detail->statut = "warn";
                        $journal_detail->description .= "&nbsp;\n";
                        $journal_detail->description .= "#### Résultat : " . "\n";
                        $journal_detail->description .= "➤ Au moins une des conditions n'a pas été validée.". "\n";
                        $journal_detail->description .= "➤ La valeur de ce cas ne sera pas prise en compte.". "\n";
                        $toutesConditionsValides = false;
                        break;
                    }

                } // fin boucle sur les conditions


                if($toutesConditionsValides) {
                    $journal_detail->reponse = "Oui";


                    $journal_detail->description .= "&nbsp;\n";
                    $journal_detail->description .= "#### Résultat : " . "\n";

                    $journal_detail->description .= "➤ Toutes les conditions ont été validées" . "\n";
                    $journal_detail->description .= "➤ La valeur choisie est donc " . $formule->valeur;

                    
                    $formuleValide = $formule;
                    $champs_dossier[$champ['description']][$nom_technique] = $formule->valeur;


                    $arrayJournalDetail[] = $journal_detail;

                    
                    break;
                }
                $arrayJournalDetail[] = $journal_detail;

            }// fin boucle sur les formules


            foreach($arrayJournalDetail as $detail){
                $detail->description .= "&nbsp;\n";
                $detail->save();
                $journal->ops_journal_detail_ops_journal->add($detail);
            }


            if(is_null($formuleValide)) {
                $journal->description .= "Aucun cas valide : la valeur par défaut du champ a été prise en compte.";
                $champs_dossier[$champ['description']][$nom_technique] = "";
            }
            else {
                $journal->description .= "Le cas '" . $formuleValide->nomDuCas . "' a été pris en compte avec la valeur : " . $formuleValide->valeur;

            }



            $journal->save();   



        } // fin boucle sur les champs grille



        return $champs_dossier;
    }
    /**
     * @access private
     * @name cleanQueryActionLot()
     * Fonction qui permet de supprimer les relations inutiles
     *
     *  @param string						    $query : requête sql
     *  @param array						    $relations : nom des relations a supprimer
     *  @return string				            $query : requête sql
     */
    public static function cleanQueryActionLot($query,$relations){
        foreach ($relations as $relation) {
            if (preg_match('/'.$relation.' jt(\d+)/', $query, $matches)) 
            {
                $numero = $matches[1];
                $where = explode('WHERE', $query);

                if(strpos($where[1],'jt'.$numero) == false && strpos($where[1],'jtl'.$numero) == false){
                    $query = str_replace(
                        'LEFT JOIN  '.$relation.' jt' . $numero . ' ON jt' . $numero . '.id=jtl' . $numero . '.'.$relation.'_id',
                        '',
                        $query
                    );
                    $query = str_replace(
                        'AND jt' . $numero . '.deleted=0',
                        '',
                        $query
                    );

                    $query = str_replace(
                        'LEFT JOIN  '.$relation.'_ops_dossier jtl' . $numero . ' ON ops_dossier.id=jtl' . $numero . '.ops_dossier_id AND jtl' . $numero . '.deleted=0',
                        '',
                        $query
                    );
                }
            }
        }
        return $query;
    }
}